"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.prepareSearchParams = void 0;
var _utils = require("../../../../../rule_management/logic/utils");
var _constants = require("../../../../../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * helper methods to prepare search params for bulk actions based on results of previous dry run
 * It excludes failed rules from search and perform bulk action on possible successfully edited rules
 * @param dryRunResult {@link DryRunResult} result of API _bulk_action dry_run
 * @param {string[]} selectedRuleIds - list of selected rule ids
 * @param filterOptions {@link FilterOptions} find filter
 * @returns either list of ids or KQL search query
 */
const prepareSearchParams = ({
  dryRunResult,
  ...props
}) => {
  // if selectedRuleIds present, filter out rules that failed during dry run
  if ('selectedRuleIds' in props) {
    const failedRuleIdsSet = new Set(dryRunResult === null || dryRunResult === void 0 ? void 0 : dryRunResult.ruleErrors.flatMap(({
      ruleIds
    }) => ruleIds));
    return {
      ids: props.selectedRuleIds.filter(id => !failedRuleIdsSet.has(id))
    };
  }

  // otherwise create filter that excludes failed results based on dry run errors
  let modifiedFilterOptions = {
    ...props.filterOptions
  };
  dryRunResult === null || dryRunResult === void 0 ? void 0 : dryRunResult.ruleErrors.forEach(({
    errorCode
  }) => {
    var _modifiedFilterOption;
    switch (errorCode) {
      case _constants.BulkActionsDryRunErrCode.IMMUTABLE:
        modifiedFilterOptions = {
          ...modifiedFilterOptions,
          showCustomRules: true
        };
        break;
      case _constants.BulkActionsDryRunErrCode.MACHINE_LEARNING_INDEX_PATTERN:
      case _constants.BulkActionsDryRunErrCode.MACHINE_LEARNING_AUTH:
        modifiedFilterOptions = {
          ...modifiedFilterOptions,
          excludeRuleTypes: [...((_modifiedFilterOption = modifiedFilterOptions.excludeRuleTypes) !== null && _modifiedFilterOption !== void 0 ? _modifiedFilterOption : []), 'machine_learning']
        };
        break;
    }
  });
  return {
    query: (0, _utils.convertRulesFilterToKQL)(modifiedFilterOptions)
  };
};
exports.prepareSearchParams = prepareSearchParams;