"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useEnableDataFeed = void 0;
var _fp = require("lodash/fp");
var _react = require("react");
var _use_app_toasts = require("../../../hooks/use_app_toasts");
var _kibana = require("../../../lib/kibana");
var _telemetry = require("../../../lib/telemetry");
var _api = require("../api");
var i18n = _interopRequireWildcard(require("./translations"));
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Enable/Disable Job & Datafeed
const useEnableDataFeed = () => {
  const {
    telemetry
  } = (0, _kibana.useKibana)().services;
  const {
    addError
  } = (0, _use_app_toasts.useAppToasts)();
  const [isLoading, setIsLoading] = (0, _react.useState)(false);
  const enableDatafeed = (0, _react.useCallback)(async (job, latestTimestampMs) => {
    setIsLoading(true);
    (0, _telemetry.track)(_telemetry.METRIC_TYPE.COUNT, job.isElasticJob ? _telemetry.TELEMETRY_EVENT.SIEM_JOB_ENABLED : _telemetry.TELEMETRY_EVENT.CUSTOM_JOB_ENABLED);
    if (!job.isInstalled) {
      try {
        await (0, _api.setupMlJob)({
          configTemplate: job.moduleId,
          indexPatternName: job.defaultIndexPattern,
          jobIdErrorFilter: [job.id],
          groups: job.groups
        });
        telemetry.reportMLJobUpdate({
          jobId: job.id,
          isElasticJob: job.isElasticJob,
          moduleId: job.moduleId,
          status: _telemetry.ML_JOB_TELEMETRY_STATUS.moduleInstalled
        });
      } catch (error) {
        setIsLoading(false);
        addError(error, {
          title: i18n.CREATE_JOB_FAILURE
        });
        telemetry.reportMLJobUpdate({
          jobId: job.id,
          isElasticJob: job.isElasticJob,
          moduleId: job.moduleId,
          status: _telemetry.ML_JOB_TELEMETRY_STATUS.installationError,
          errorMessage: `${i18n.CREATE_JOB_FAILURE} - ${error.message}`
        });
        return {
          enabled: false
        };
      }
    }

    // Max start time for job is no more than two weeks ago to ensure job performance
    const date = new Date();
    const maxStartTime = date.setDate(date.getDate() - 14);
    const datafeedId = `datafeed-${job.id}`;
    const startTime = Math.max(latestTimestampMs, maxStartTime);
    try {
      var _response$datafeedId;
      const response = await (0, _api.startDatafeeds)({
        datafeedIds: [datafeedId],
        start: startTime
      });
      if ((_response$datafeedId = response[datafeedId]) !== null && _response$datafeedId !== void 0 && _response$datafeedId.error) {
        throw new Error(response[datafeedId].error);
      }
      telemetry.reportMLJobUpdate({
        jobId: job.id,
        isElasticJob: job.isElasticJob,
        status: _telemetry.ML_JOB_TELEMETRY_STATUS.started
      });
      return {
        enabled: response[datafeedId] ? response[datafeedId].started : false
      };
    } catch (error) {
      (0, _telemetry.track)(_telemetry.METRIC_TYPE.COUNT, _telemetry.TELEMETRY_EVENT.JOB_ENABLE_FAILURE);
      addError(error, {
        title: i18n.START_JOB_FAILURE
      });
      telemetry.reportMLJobUpdate({
        jobId: job.id,
        isElasticJob: job.isElasticJob,
        status: _telemetry.ML_JOB_TELEMETRY_STATUS.startError,
        errorMessage: `${i18n.START_JOB_FAILURE} - ${error.message}`
      });
    } finally {
      setIsLoading(false);
    }
    return {
      enabled: false
    };
  }, [addError, telemetry]);
  const disableDatafeed = (0, _react.useCallback)(async job => {
    (0, _telemetry.track)(_telemetry.METRIC_TYPE.COUNT, job.isElasticJob ? _telemetry.TELEMETRY_EVENT.SIEM_JOB_DISABLED : _telemetry.TELEMETRY_EVENT.CUSTOM_JOB_DISABLED);
    setIsLoading(true);
    const datafeedId = `datafeed-${job.id}`;
    try {
      const [response] = await (0, _api.stopDatafeeds)({
        datafeedIds: [datafeedId]
      });
      if (isErrorResponse(response)) {
        throw new Error(response.error);
      }
      telemetry.reportMLJobUpdate({
        jobId: job.id,
        isElasticJob: job.isElasticJob,
        status: _telemetry.ML_JOB_TELEMETRY_STATUS.stopped
      });
      return {
        enabled: response[datafeedId] ? !response[datafeedId].stopped : true
      };
    } catch (error) {
      (0, _telemetry.track)(_telemetry.METRIC_TYPE.COUNT, _telemetry.TELEMETRY_EVENT.JOB_DISABLE_FAILURE);
      addError(error, {
        title: i18n.STOP_JOB_FAILURE
      });
      telemetry.reportMLJobUpdate({
        jobId: job.id,
        isElasticJob: job.isElasticJob,
        status: _telemetry.ML_JOB_TELEMETRY_STATUS.stopError,
        errorMessage: `${i18n.STOP_JOB_FAILURE} - ${error.message}`
      });
    } finally {
      setIsLoading(false);
    }
    return {
      enabled: true
    };
  }, [addError, telemetry]);
  return {
    enableDatafeed,
    disableDatafeed,
    isLoading
  };
};
exports.useEnableDataFeed = useEnableDataFeed;
const isErrorResponse = response => !(0, _fp.isEmpty)(response.error);