"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SecurityCheckupService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _operators = require("rxjs/operators");
var _components = require("./components");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_SECURITY_CHECKUP_STATE = Object.freeze({
  displayAlert: false
});
class SecurityCheckupService {
  constructor(config, storage) {
    (0, _defineProperty2.default)(this, "enabled", void 0);
    (0, _defineProperty2.default)(this, "alertVisibility$", void 0);
    (0, _defineProperty2.default)(this, "storage", void 0);
    (0, _defineProperty2.default)(this, "alertToast", void 0);
    (0, _defineProperty2.default)(this, "storageKey", void 0);
    this.storage = storage;
    this.enabled = config.showInsecureClusterWarning;
    this.alertVisibility$ = new _rxjs.BehaviorSubject(this.enabled);
  }
  setup({
    http
  }) {
    const tenant = http.basePath.serverBasePath;
    this.storageKey = `insecureClusterWarningVisibility${tenant}`;
    this.enabled = this.enabled && this.getPersistedVisibilityPreference();
    this.alertVisibility$.next(this.enabled);
  }
  start(startDeps) {
    if (this.enabled) {
      this.initializeAlert(startDeps);
    }
  }
  initializeAlert({
    http,
    notifications,
    docLinks
  }) {
    const appState$ = (0, _rxjs.from)(this.getSecurityCheckupState(http));

    // 10 days is reasonably long enough to call "forever" for a page load.
    // Can't go too much longer than this. See https://github.com/elastic/kibana/issues/64264#issuecomment-618400354
    const oneMinute = 60000;
    const tenDays = oneMinute * 60 * 24 * 10;
    (0, _rxjs.combineLatest)([appState$, this.alertVisibility$]).pipe((0, _operators.map)(([{
      displayAlert
    }, isAlertVisible]) => displayAlert && isAlertVisible), (0, _operators.distinctUntilChanged)()).subscribe(showAlert => {
      if (showAlert && !this.alertToast) {
        this.alertToast = notifications.toasts.addWarning({
          title: _components.insecureClusterAlertTitle,
          text: (0, _components.insecureClusterAlertText)(docLinks, persist => this.setAlertVisibility(false, persist)),
          iconType: 'warning'
        }, {
          toastLifeTimeMs: tenDays
        });
      } else if (!showAlert && this.alertToast) {
        notifications.toasts.remove(this.alertToast);
        this.alertToast = undefined;
      }
    });
  }
  getSecurityCheckupState(http) {
    return http.anonymousPaths.isAnonymous(window.location.pathname) ? Promise.resolve(DEFAULT_SECURITY_CHECKUP_STATE) : http.get('/internal/security/security_checkup/state').catch(() => DEFAULT_SECURITY_CHECKUP_STATE);
  }
  setAlertVisibility(show, persist) {
    if (!this.enabled) {
      return;
    }
    this.alertVisibility$.next(show);
    if (persist) {
      this.setPersistedVisibilityPreference(show);
    }
  }
  getPersistedVisibilityPreference() {
    var _this$storage$getItem;
    const entry = (_this$storage$getItem = this.storage.getItem(this.storageKey)) !== null && _this$storage$getItem !== void 0 ? _this$storage$getItem : '{}';
    try {
      const {
        show = true
      } = JSON.parse(entry);
      return show;
    } catch (e) {
      return true;
    }
  }
  setPersistedVisibilityPreference(show) {
    this.storage.setItem(this.storageKey, JSON.stringify({
      show
    }));
  }
}
exports.SecurityCheckupService = SecurityCheckupService;