"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useFormChanges = exports.FormChangesProvider = void 0;
exports.useFormChangesContext = useFormChangesContext;
var _react = require("react");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Custom React hook that allows tracking changes within a form.
 *
 * @example
 * ```
 * const { count } = useFormChanges(); // Form has {count} unsaved changes
 * ```
 */
const useFormChanges = () => {
  const [count, setCount] = (0, _react.useState)(0);
  return {
    count,
    report: isEqual => {
      if (!isEqual) {
        setCount(c => c + 1);
        return () => setCount(c => c - 1);
      }
    }
  };
};
exports.useFormChanges = useFormChanges;
const FormChangesContext = /*#__PURE__*/(0, _react.createContext)(undefined);
const FormChangesProvider = FormChangesContext.Provider;

/**
 * Custom React hook that returns all @see FormChangesProps state from context.
 *
 * @throws Error if called within a component that isn't a child of a `<FormChanges>` component.
 */
exports.FormChangesProvider = FormChangesProvider;
function useFormChangesContext() {
  const value = (0, _react.useContext)(FormChangesContext);
  if (!value) {
    throw new Error('FormChanges context is undefined, please verify you are calling useFormChangesContext() as child of a <FormChanges> component.');
  }
  return value;
}