"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerSetupRoute = registerSetupRoute;
var _common = require("@kbn/spaces-plugin/common");
var _compat = require("./compat");
var _apm_package = require("../lib/setup/apm_package");
var _cluster_settings = require("../lib/setup/cluster_settings");
var _fleet_policies = require("../lib/setup/fleet_policies");
var _get_setup_instructions = require("../lib/setup/get_setup_instructions");
var _has_profiling_data = require("../lib/setup/has_profiling_data");
var _security_role = require("../lib/setup/security_role");
var _handle_route_error_handler = require("../utils/handle_route_error_handler");
var _common2 = require("../../common");
var _setup = require("../../common/setup");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerSetupRoute({
  router,
  logger,
  services: {
    createProfilingEsClient
  },
  dependencies
}) {
  const paths = (0, _common2.getRoutePaths)();
  // Check if Elasticsearch and Fleet are setup for Universal Profiling
  router.get({
    path: paths.HasSetupESResources,
    options: {
      tags: ['access:profiling']
    },
    validate: false
  }, async (context, request, response) => {
    try {
      var _dependencies$setup$s, _dependencies$setup$s2, _dependencies$setup$s3;
      const esClient = await (0, _compat.getClient)(context);
      const core = await context.core;
      const clientWithDefaultAuth = createProfilingEsClient({
        esClient,
        request,
        useDefaultAuth: true
      });
      const clientWithProfilingAuth = createProfilingEsClient({
        esClient,
        request,
        useDefaultAuth: false
      });
      const setupOptions = {
        client: clientWithDefaultAuth,
        logger,
        packagePolicyClient: dependencies.start.fleet.packagePolicyService,
        soClient: core.savedObjects.client,
        spaceId: (_dependencies$setup$s = (_dependencies$setup$s2 = dependencies.setup.spaces) === null || _dependencies$setup$s2 === void 0 ? void 0 : (_dependencies$setup$s3 = _dependencies$setup$s2.spacesService) === null || _dependencies$setup$s3 === void 0 ? void 0 : _dependencies$setup$s3.getSpaceId(request)) !== null && _dependencies$setup$s !== void 0 ? _dependencies$setup$s : _common.DEFAULT_SPACE_ID,
        isCloudEnabled: dependencies.setup.cloud.isCloudEnabled,
        config: dependencies.config
      };
      const state = (0, _setup.createDefaultSetupState)();
      state.cloud.available = dependencies.setup.cloud.isCloudEnabled;
      if (!state.cloud.available) {
        const msg = `Elastic Cloud is required to set up Elasticsearch and Fleet for Universal Profiling`;
        logger.error(msg);
        return response.custom({
          statusCode: 500,
          body: {
            message: msg
          }
        });
      }
      state.data.available = await (0, _has_profiling_data.hasProfilingData)({
        ...setupOptions,
        client: clientWithProfilingAuth
      });
      if (state.data.available) {
        return response.ok({
          body: {
            has_setup: true,
            has_data: state.data.available
          }
        });
      }
      const verifyFunctions = [_apm_package.isApmPackageInstalled, _fleet_policies.validateApmPolicy, _fleet_policies.validateCollectorPackagePolicy, _cluster_settings.validateMaximumBuckets, _cluster_settings.validateResourceManagement, _security_role.validateSecurityRole, _fleet_policies.validateSymbolizerPackagePolicy];
      const partialStates = await Promise.all(verifyFunctions.map(fn => fn(setupOptions)));
      const mergedState = (0, _setup.mergePartialSetupStates)(state, partialStates);
      return response.ok({
        body: {
          has_setup: (0, _setup.areResourcesSetup)(mergedState),
          has_data: mergedState.data.available
        }
      });
    } catch (error) {
      return (0, _handle_route_error_handler.handleRouteHandlerError)({
        error,
        logger,
        response,
        message: 'Error while checking plugin setup'
      });
    }
  });
  // Set up Elasticsearch and Fleet for Universal Profiling
  router.post({
    path: paths.HasSetupESResources,
    options: {
      tags: ['access:profiling']
    },
    validate: false
  }, async (context, request, response) => {
    try {
      var _dependencies$setup$s4, _dependencies$setup$s5, _dependencies$setup$s6;
      const esClient = await (0, _compat.getClient)(context);
      const core = await context.core;
      const clientWithDefaultAuth = createProfilingEsClient({
        esClient,
        request,
        useDefaultAuth: true
      });
      const setupOptions = {
        client: clientWithDefaultAuth,
        logger,
        packagePolicyClient: dependencies.start.fleet.packagePolicyService,
        soClient: core.savedObjects.client,
        spaceId: (_dependencies$setup$s4 = (_dependencies$setup$s5 = dependencies.setup.spaces) === null || _dependencies$setup$s5 === void 0 ? void 0 : (_dependencies$setup$s6 = _dependencies$setup$s5.spacesService) === null || _dependencies$setup$s6 === void 0 ? void 0 : _dependencies$setup$s6.getSpaceId(request)) !== null && _dependencies$setup$s4 !== void 0 ? _dependencies$setup$s4 : _common.DEFAULT_SPACE_ID,
        isCloudEnabled: dependencies.setup.cloud.isCloudEnabled,
        config: dependencies.config
      };
      const state = (0, _setup.createDefaultSetupState)();
      state.cloud.available = dependencies.setup.cloud.isCloudEnabled;
      if (!state.cloud.available) {
        const msg = `Elastic Cloud is required to set up Elasticsearch and Fleet for Universal Profiling`;
        logger.error(msg);
        return response.custom({
          statusCode: 500,
          body: {
            message: msg
          }
        });
      }
      const partialStates = await Promise.all([_apm_package.isApmPackageInstalled, _fleet_policies.validateApmPolicy, _fleet_policies.validateCollectorPackagePolicy, _cluster_settings.validateMaximumBuckets, _cluster_settings.validateResourceManagement, _security_role.validateSecurityRole, _fleet_policies.validateSymbolizerPackagePolicy].map(fn => fn(setupOptions)));
      const mergedState = (0, _setup.mergePartialSetupStates)(state, partialStates);
      const executeFunctions = [...(mergedState.packages.installed ? [] : [_apm_package.installLatestApmPackage]), ...(mergedState.policies.apm.installed ? [] : [_fleet_policies.updateApmPolicy]), ...(mergedState.policies.collector.installed ? [] : [_fleet_policies.createCollectorPackagePolicy]), ...(mergedState.policies.symbolizer.installed ? [] : [_fleet_policies.createSymbolizerPackagePolicy]), ...(mergedState.resource_management.enabled ? [] : [_cluster_settings.enableResourceManagement]), ...(mergedState.permissions.configured ? [] : [_security_role.setSecurityRole]), ...(mergedState.settings.configured ? [] : [_cluster_settings.setMaximumBuckets])];
      if (!executeFunctions.length) {
        return response.ok();
      }
      await Promise.all(executeFunctions.map(fn => fn(setupOptions)));

      // We return a status code of 202 instead of 200 because enabling
      // resource management in Elasticsearch is an asynchronous action
      // and is not guaranteed to complete before Kibana sends a response.
      return response.accepted();
    } catch (error) {
      return (0, _handle_route_error_handler.handleRouteHandlerError)({
        error,
        logger,
        response,
        message: 'Error while setting up Universal Profiling'
      });
    }
  });
  // Show users the instructions on how to setup Universal Profiling agents
  router.get({
    path: paths.SetupDataCollectionInstructions,
    options: {
      tags: ['access:profiling']
    },
    validate: false
  }, async (context, request, response) => {
    try {
      var _dependencies$setup$c, _dependencies$setup$c2;
      const apmServerHost = (_dependencies$setup$c = dependencies.setup.cloud) === null || _dependencies$setup$c === void 0 ? void 0 : (_dependencies$setup$c2 = _dependencies$setup$c.apm) === null || _dependencies$setup$c2 === void 0 ? void 0 : _dependencies$setup$c2.url;
      const setupInstructions = await (0, _get_setup_instructions.getSetupInstructions)({
        packagePolicyClient: dependencies.start.fleet.packagePolicyService,
        soClient: (await context.core).savedObjects.client,
        apmServerHost
      });
      return response.ok({
        body: setupInstructions
      });
    } catch (error) {
      return (0, _handle_route_error_handler.handleRouteHandlerError)({
        error,
        logger,
        response,
        message: 'Error while fetching Universal Profiling instructions'
      });
    }
  });
}