"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.decodeStackTraceResponse = decodeStackTraceResponse;
exports.makeFrameID = void 0;
exports.searchStackTraces = searchStackTraces;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const makeFrameID = (frameID, n) => {
  return n === 0 ? frameID : frameID + ';' + n.toString();
};

// createInlineTrace builds a new StackTrace with inline frames.
exports.makeFrameID = makeFrameID;
const createInlineTrace = (trace, frames) => {
  // The arrays need to be extended with the inline frame information.
  const frameIDs = [];
  const fileIDs = [];
  const addressOrLines = [];
  const typeIDs = [];
  for (let i = 0; i < trace.frame_ids.length; i++) {
    const frameID = trace.frame_ids[i];
    frameIDs.push(frameID);
    fileIDs.push(trace.file_ids[i]);
    addressOrLines.push(trace.address_or_lines[i]);
    typeIDs.push(trace.type_ids[i]);
    for (let j = 1;; j++) {
      const inlineID = makeFrameID(frameID, j);
      const frame = frames.get(inlineID);
      if (!frame) {
        break;
      }
      frameIDs.push(inlineID);
      fileIDs.push(trace.file_ids[i]);
      addressOrLines.push(trace.address_or_lines[i]);
      typeIDs.push(trace.type_ids[i]);
    }
  }
  return {
    FrameIDs: frameIDs,
    FileIDs: fileIDs,
    AddressOrLines: addressOrLines,
    Types: typeIDs
  };
};
function decodeStackTraceResponse(response) {
  const stackTraceEvents = new Map();
  for (const [key, value] of Object.entries((_response$stack_trace = response.stack_trace_events) !== null && _response$stack_trace !== void 0 ? _response$stack_trace : {})) {
    var _response$stack_trace;
    stackTraceEvents.set(key, value);
  }
  const stackFrames = new Map();
  for (const [frameID, frame] of Object.entries((_response$stack_frame = response.stack_frames) !== null && _response$stack_frame !== void 0 ? _response$stack_frame : {})) {
    var _response$stack_frame;
    // Each field in a stackframe is represented by an array. This is
    // necessary to support inline frames.
    //
    // We store the inlined frames with a modified (and unique) ID.
    // We can do so since we don't display the frame IDs.
    for (let i = 0; i < frame.function_name.length; i++) {
      stackFrames.set(makeFrameID(frameID, i), {
        FileName: frame.file_name[i],
        FunctionName: frame.function_name[i],
        FunctionOffset: frame.function_offset[i],
        LineNumber: frame.line_number[i],
        Inline: i > 0
      });
    }
  }
  const stackTraces = new Map();
  for (const [traceID, trace] of Object.entries((_response$stack_trace2 = response.stack_traces) !== null && _response$stack_trace2 !== void 0 ? _response$stack_trace2 : {})) {
    var _response$stack_trace2;
    stackTraces.set(traceID, createInlineTrace(trace, stackFrames));
  }
  const executables = new Map();
  for (const [key, value] of Object.entries((_response$executables = response.executables) !== null && _response$executables !== void 0 ? _response$executables : {})) {
    var _response$executables;
    executables.set(key, {
      FileName: value
    });
  }
  return {
    stackTraceEvents,
    stackTraces,
    stackFrames,
    executables,
    totalFrames: response.total_frames,
    samplingRate: response.sampling_rate
  };
}
async function searchStackTraces({
  client,
  filter,
  sampleSize
}) {
  const response = await client.profilingStacktraces({
    query: filter,
    sampleSize
  });
  return decodeStackTraceResponse(response);
}