"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createCollectorPackagePolicy = createCollectorPackagePolicy;
exports.createSymbolizerPackagePolicy = createSymbolizerPackagePolicy;
exports.generateSecretToken = generateSecretToken;
exports.getCollectorPolicy = getCollectorPolicy;
exports.getSymbolizerPolicy = getSymbolizerPolicy;
exports.getVarsFor = getVarsFor;
exports.updateApmPolicy = updateApmPolicy;
exports.validateApmPolicy = validateApmPolicy;
exports.validateCollectorPackagePolicy = validateCollectorPackagePolicy;
exports.validateSymbolizerPackagePolicy = validateSymbolizerPackagePolicy;
var _lodash = require("lodash");
var _registry = require("@kbn/fleet-plugin/server/services/epm/registry");
var _get_apm_policy = require("./get_apm_policy");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function createIngestAPIKey(esClient) {
  const apiKeyResponse = await esClient.security.createApiKey({
    name: 'profiling-manager',
    role_descriptors: {
      profiling_manager: {
        indices: [{
          names: ['profiling-*', '.profiling-*'],
          privileges: ['read', 'create_doc', 'create', 'write', 'index', 'create_index', 'view_index_metadata', 'manage']
        }],
        cluster: ['monitor']
      }
    }
  });
  return atob(apiKeyResponse.encoded);
}
async function validateApmPolicy({
  soClient,
  packagePolicyClient
}) {
  var _apmPolicy$inputs$0$c, _apmPolicy$inputs$0$c2;
  const apmPolicy = await (0, _get_apm_policy.getApmPolicy)({
    packagePolicyClient,
    soClient
  });
  return {
    policies: {
      apm: {
        installed: !!(apmPolicy && apmPolicy !== null && apmPolicy !== void 0 && (_apmPolicy$inputs$0$c = apmPolicy.inputs[0].config) !== null && _apmPolicy$inputs$0$c !== void 0 && (_apmPolicy$inputs$0$c2 = _apmPolicy$inputs$0$c['apm-server'].value) !== null && _apmPolicy$inputs$0$c2 !== void 0 && _apmPolicy$inputs$0$c2.profiling)
      }
    }
  };
}
async function updateApmPolicy({
  client,
  soClient,
  packagePolicyClient
}) {
  const esClient = client.getEsClient();
  const apmPolicy = await (0, _get_apm_policy.getApmPolicy)({
    packagePolicyClient,
    soClient
  });
  if (!apmPolicy) {
    throw new Error(`Could not find APM policy`);
  }
  const apmPolicyApiKey = await createIngestAPIKey(esClient);
  const profilingApmConfig = {
    profiling: {
      enabled: true,
      elasticsearch: {
        api_key: apmPolicyApiKey
      },
      metrics: {
        elasticsearch: {
          hosts: ['https://1b6c02856ea642a6ac14499b01507233.us-east-2.aws.elastic-cloud.com:443'],
          api_key: 'woq-IoMBRbbiEbPugtWW:_iBmc1PdSout7sf5FCkEpA'
        }
      },
      keyvalue_retention: {
        // 60 days
        age: '1440h',
        // 200 Gib
        size_bytes: 200 * 1024 * 1024 * 1024,
        execution_interval: '12h'
      }
    }
  };
  const {
    id,
    revision,
    updated_at: updateAt,
    updated_by: updateBy,
    ...apmPolicyModified
  } = apmPolicy;
  apmPolicyModified.inputs = apmPolicy.inputs.map(input => {
    return input.type === 'apm' ? (0, _lodash.merge)({}, input, {
      config: {
        'apm-server': {
          value: profilingApmConfig
        }
      }
    }) : input;
  });
  await packagePolicyClient.update(soClient, esClient, id, apmPolicyModified);
}
const CLOUD_AGENT_POLICY_ID = 'policy-elastic-agent-on-cloud';
const COLLECTOR_PACKAGE_POLICY_NAME = 'elastic-universal-profiling-collector';
const SYMBOLIZER_PACKAGE_POLICY_NAME = 'elastic-universal-profiling-symbolizer';
async function getPackagePolicy({
  soClient,
  packagePolicyClient,
  packageName
}) {
  const packagePolicies = await packagePolicyClient.list(soClient, {});
  return packagePolicies.items.find(pkg => pkg.name === packageName);
}
async function getCollectorPolicy({
  soClient,
  packagePolicyClient
}) {
  return getPackagePolicy({
    soClient,
    packagePolicyClient,
    packageName: COLLECTOR_PACKAGE_POLICY_NAME
  });
}
async function validateCollectorPackagePolicy({
  soClient,
  packagePolicyClient
}) {
  const collectorPolicy = await getCollectorPolicy({
    soClient,
    packagePolicyClient
  });
  return {
    policies: {
      collector: {
        installed: !!collectorPolicy
      }
    }
  };
}
function generateSecretToken() {
  let result = '';
  const characters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789';
  for (let i = 0; i < 16; i++) {
    const randomIndex = Math.floor(Math.random() * characters.length);
    result += characters.charAt(randomIndex);
  }
  return result;
}
function getVarsFor({
  config,
  includeSecretToken
}) {
  const configKeys = Object.keys(config);
  if (includeSecretToken) {
    configKeys.push('secret_token');
  }
  return configKeys.reduce((acc, currKey) => {
    const value = currKey === 'secret_token' ? generateSecretToken() : config[currKey];
    const type = typeof value === 'boolean' ? 'bool' : 'text';
    return {
      ...acc,
      [currKey]: {
        type,
        value
      }
    };
  }, {});
}
async function createCollectorPackagePolicy({
  client,
  soClient,
  packagePolicyClient,
  config
}) {
  const packageName = 'profiler_collector';
  const {
    version
  } = await (0, _registry.fetchFindLatestPackageOrThrow)(packageName, {
    prerelease: true
  });
  const packagePolicy = {
    policy_id: CLOUD_AGENT_POLICY_ID,
    enabled: true,
    package: {
      name: packageName,
      title: 'Universal Profiling Collector',
      version
    },
    name: COLLECTOR_PACKAGE_POLICY_NAME,
    namespace: 'default',
    inputs: [{
      policy_template: 'universal_profiling_collector',
      enabled: true,
      streams: [],
      type: 'pf-elastic-collector',
      vars: config !== null && config !== void 0 && config.collector ? getVarsFor({
        config: config.collector,
        includeSecretToken: true
      }) : {}
    }]
  };
  const esClient = client.getEsClient();
  await packagePolicyClient.create(soClient, esClient, packagePolicy, {
    force: true
  });
}
async function getSymbolizerPolicy({
  soClient,
  packagePolicyClient
}) {
  return getPackagePolicy({
    soClient,
    packagePolicyClient,
    packageName: SYMBOLIZER_PACKAGE_POLICY_NAME
  });
}
async function validateSymbolizerPackagePolicy({
  soClient,
  packagePolicyClient
}) {
  const symbolizerPackagePolicy = await getSymbolizerPolicy({
    soClient,
    packagePolicyClient
  });
  return {
    policies: {
      symbolizer: {
        installed: !!symbolizerPackagePolicy
      }
    }
  };
}
async function createSymbolizerPackagePolicy({
  client,
  soClient,
  packagePolicyClient,
  config
}) {
  const packageName = 'profiler_symbolizer';
  const {
    version
  } = await (0, _registry.fetchFindLatestPackageOrThrow)(packageName, {
    prerelease: true
  });
  const packagePolicy = {
    policy_id: CLOUD_AGENT_POLICY_ID,
    enabled: true,
    package: {
      name: packageName,
      title: 'Universal Profiling Symbolizer',
      version
    },
    name: SYMBOLIZER_PACKAGE_POLICY_NAME,
    namespace: 'default',
    inputs: [{
      policy_template: 'universal_profiling_symbolizer',
      enabled: true,
      streams: [],
      type: 'pf-elastic-symbolizer',
      // doesnt have secret token
      vars: config !== null && config !== void 0 && config.symbolizer ? getVarsFor({
        config: config.symbolizer,
        includeSecretToken: false
      }) : {}
    }]
  };
  const esClient = client.getEsClient();
  await packagePolicyClient.create(soClient, esClient, packagePolicy, {
    force: true
  });
}