"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateCompositeSLO = validateCompositeSLO;
var _sloSchema = require("@kbn/slo-schema");
var _errors = require("../../../errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function validateCompositeSLO(compositeSlo, sloList) {
  assertNumberOfSourceSlo(compositeSlo);
  assertMatchingSloList(compositeSlo, sloList);
  assertSameBudgetingMethod(compositeSlo, sloList);
  assertSameTimeWindow(compositeSlo, sloList);
}
function assertNumberOfSourceSlo(compositeSlo) {
  if (compositeSlo.sources.length < 2 || compositeSlo.sources.length > 30) {
    throw new _errors.IllegalArgumentError('A composite SLO must contain between 2 and 30 source SLOs.');
  }
}
function assertMatchingSloList(compositeSlo, sloList) {
  const everySourceSloMatches = compositeSlo.sources.every(sourceSlo => sloList.find(slo => sourceSlo.id === slo.id && sourceSlo.revision === slo.revision));
  if (!everySourceSloMatches) {
    throw new _errors.IllegalArgumentError('One or many source SLOs are not matching the specified id and revision.');
  }
}
function assertSameBudgetingMethod(compositeSlo, sloList) {
  const haveSameBudgetingMethod = sloList.every(slo => slo.budgetingMethod === compositeSlo.budgetingMethod);
  if (compositeSlo.budgetingMethod === 'timeslices') {
    if (compositeSlo.objective.timesliceWindow === undefined) {
      throw new _errors.IllegalArgumentError('Invalid timeslices objective. A timeslice window must be set and equal to all source SLO.');
    }
    const haveSameTimesliceWindow = sloList.every(slo => {
      var _slo$objective$timesl;
      return (_slo$objective$timesl = slo.objective.timesliceWindow) === null || _slo$objective$timesl === void 0 ? void 0 : _slo$objective$timesl.isEqual(compositeSlo.objective.timesliceWindow);
    });
    if (!haveSameTimesliceWindow) {
      throw new _errors.IllegalArgumentError('Invalid budgeting method. Every source SLO must use the same timeslice window.');
    }
  }
  if (!haveSameBudgetingMethod) {
    throw new _errors.IllegalArgumentError('Invalid budgeting method. Every source SLO must use the same budgeting method as the composite.');
  }
}
function assertSameTimeWindow(compositeSlo, sloList) {
  let haveSameTimeWindow = false;
  if (_sloSchema.rollingTimeWindowSchema.is(compositeSlo.timeWindow)) {
    haveSameTimeWindow = sloList.every(slo => slo.timeWindow.duration.isEqual(compositeSlo.timeWindow.duration) && _sloSchema.rollingTimeWindowSchema.is(slo.timeWindow));
  }
  if (_sloSchema.calendarAlignedTimeWindowSchema.is(compositeSlo.timeWindow)) {
    haveSameTimeWindow = sloList.every(slo => slo.timeWindow.duration.isEqual(compositeSlo.timeWindow.duration) && _sloSchema.calendarAlignedTimeWindowSchema.is(slo.timeWindow));
  }
  if (!haveSameTimeWindow) {
    throw new _errors.IllegalArgumentError('Invalid time window. Every source SLO must use the same time window as the composite.');
  }
}