"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertDataViewIntoLensIndexPattern = convertDataViewIntoLensIndexPattern;
exports.ensureIndexPattern = ensureIndexPattern;
exports.getFieldByNameFactory = getFieldByNameFactory;
exports.loadIndexPatternRefs = loadIndexPatternRefs;
exports.loadIndexPatterns = loadIndexPatterns;
var _public = require("@kbn/visualization-ui-components/public");
var _lodash = require("lodash");
var _document_field = require("../datasources/form_based/document_field");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * All these functions will be used by the Embeddable instance too,
 * therefore keep all these functions pretty raw here and do not use the IndexPatternService
 */

function getFieldByNameFactory(newFields) {
  const fieldsLookup = (0, _lodash.keyBy)(newFields, 'name');
  return name => fieldsLookup[name];
}
function convertDataViewIntoLensIndexPattern(dataView, restrictionRemapper = onRestrictionMapping) {
  const newFields = dataView.fields.filter(_public.isFieldLensCompatible).map(field => {
    // Convert the getters on the index pattern service into plain JSON
    const base = {
      name: field.name,
      displayName: field.displayName,
      type: field.type,
      aggregatable: field.aggregatable,
      filterable: field.filterable,
      searchable: field.searchable,
      meta: dataView.metaFields.includes(field.name),
      esTypes: field.esTypes,
      scripted: field.scripted,
      isMapped: field.isMapped,
      customLabel: field.customLabel,
      runtimeField: field.runtimeField,
      runtime: Boolean(field.runtimeField),
      timeSeriesMetric: field.timeSeriesMetric,
      timeSeriesRollup: field.isRolledUpField,
      partiallyApplicableFunctions: field.isRolledUpField ? {
        percentile: true,
        percentile_rank: true,
        median: true,
        last_value: true,
        unique_count: true,
        standard_deviation: true
      } : undefined
    };

    // Simplifies tests by hiding optional properties instead of undefined
    return base.scripted ? {
      ...base,
      lang: field.lang,
      script: field.script
    } : base;
  }).concat(_document_field.documentField);
  const {
    typeMeta,
    title,
    name,
    timeFieldName,
    fieldFormatMap
  } = dataView;
  if (typeMeta !== null && typeMeta !== void 0 && typeMeta.aggs) {
    const aggs = Object.keys(typeMeta.aggs);
    newFields.forEach((field, index) => {
      const restrictionsObj = {};
      aggs.forEach(agg => {
        const restriction = typeMeta.aggs && typeMeta.aggs[agg] && typeMeta.aggs[agg][field.name];
        if (restriction) {
          restrictionsObj[restrictionRemapper(agg)] = restriction;
        }
      });
      if (Object.keys(restrictionsObj).length) {
        newFields[index] = {
          ...field,
          aggregationRestrictions: restrictionsObj
        };
      }
    });
  }
  return {
    id: dataView.id,
    // id exists for sure because we got index patterns by id
    title,
    name: name ? name : title,
    timeFieldName,
    fieldFormatMap: fieldFormatMap && Object.fromEntries(Object.entries(fieldFormatMap).map(([id, format]) => [id,
    // @ts-ignore
    'toJSON' in format ? format.toJSON() : format])),
    fields: newFields,
    getFieldByName: getFieldByNameFactory(newFields),
    hasRestrictions: !!(typeMeta !== null && typeMeta !== void 0 && typeMeta.aggs),
    spec: dataView.toSpec(false),
    isPersisted: dataView.isPersisted()
  };
}
async function loadIndexPatternRefs(dataViews) {
  const indexPatternsRefs = await dataViews.getIdsWithTitle();
  return (0, _utils.sortDataViewRefs)(indexPatternsRefs);
}

/**
 * Map ES agg names with Lens ones
 */
const renameOperationsMapping = {
  avg: 'average',
  cardinality: 'unique_count'
};
function onRestrictionMapping(agg) {
  return agg in renameOperationsMapping ? renameOperationsMapping[agg] : agg;
}
async function loadIndexPatterns({
  dataViews,
  patterns,
  notUsedPatterns,
  cache,
  adHocDataViews,
  onIndexPatternRefresh
}) {
  const missingIds = patterns.filter(id => !cache[id] && !(adHocDataViews !== null && adHocDataViews !== void 0 && adHocDataViews[id]));
  const hasAdHocDataViews = Object.values(adHocDataViews || {}).length > 0;
  if (missingIds.length === 0 && !hasAdHocDataViews) {
    return cache;
  }
  onIndexPatternRefresh === null || onIndexPatternRefresh === void 0 ? void 0 : onIndexPatternRefresh();
  const allIndexPatterns = await Promise.allSettled(missingIds.map(id => dataViews.get(id)));
  // ignore rejected indexpatterns here, they're already handled at the app level
  let indexPatterns = allIndexPatterns.filter(response => response.status === 'fulfilled').map(response => response.value);

  // if all of the used index patterns failed to load, try loading one of not used ones till one succeeds
  if (!indexPatterns.length && !hasAdHocDataViews && notUsedPatterns) {
    for (const notUsedPattern of notUsedPatterns) {
      const resp = await dataViews.get(notUsedPattern).catch(e => {
        // do nothing
      });
      if (resp) {
        indexPatterns = [resp];
        break;
      }
    }
  }
  indexPatterns.push(...(await Promise.all(Object.values(adHocDataViews || {}).map(spec => dataViews.create(spec)))));
  const indexPatternsObject = indexPatterns.reduce((acc, indexPattern) => ({
    [indexPattern.id]: convertDataViewIntoLensIndexPattern(indexPattern, onRestrictionMapping),
    ...acc
  }), {
    ...cache
  });
  return indexPatternsObject;
}
async function ensureIndexPattern({
  id,
  onError,
  dataViews,
  cache = {}
}) {
  const indexPatterns = await loadIndexPatterns({
    dataViews,
    cache,
    patterns: [id]
  });
  if (indexPatterns[id] == null) {
    onError(Error('Missing indexpatterns'));
    return;
  }
  const newIndexPatterns = {
    ...cache,
    [id]: indexPatterns[id]
  };
  return newIndexPatterns;
}