"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.logEntryRateModule = void 0;
var _i18n = require("@kbn/i18n");
var _log_analysis = require("../../../../../../common/log_analysis");
var _log_analysis_cleanup = require("../../log_analysis_cleanup");
var _ml_get_jobs_summary_api = require("../../api/ml_get_jobs_summary_api");
var _ml_get_module = require("../../api/ml_get_module");
var _ml_setup_module_api = require("../../api/ml_setup_module_api");
var _validate_datasets = require("../../api/validate_datasets");
var _validate_indices = require("../../api/validate_indices");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const moduleId = 'logs_ui_analysis';
const moduleName = _i18n.i18n.translate('xpack.infra.logs.analysis.logEntryRateModuleName', {
  defaultMessage: 'Log rate'
});
const moduleDescription = _i18n.i18n.translate('xpack.infra.logs.analysis.logEntryRateModuleDescription', {
  defaultMessage: 'Use Machine Learning to automatically detect anomalous log entry rates.'
});
const getJobIds = (spaceId, logViewId) => _log_analysis.logEntryRateJobTypes.reduce((accumulatedJobIds, jobType) => ({
  ...accumulatedJobIds,
  [jobType]: (0, _log_analysis.getJobId)(spaceId, logViewId, jobType)
}), {});
const getJobSummary = async (spaceId, logViewId, fetch) => {
  const response = await (0, _ml_get_jobs_summary_api.callJobsSummaryAPI)({
    spaceId,
    logViewId,
    jobTypes: _log_analysis.logEntryRateJobTypes
  }, fetch);
  const jobIds = Object.values(getJobIds(spaceId, logViewId));
  return response.filter(jobSummary => jobIds.includes(jobSummary.id));
};
const getModuleDefinition = async fetch => {
  return await (0, _ml_get_module.callGetMlModuleAPI)(moduleId, fetch);
};
const setUpModule = async (start, end, datasetFilter, {
  spaceId,
  sourceId,
  indices,
  timestampField,
  runtimeMappings
}, fetch) => {
  const indexNamePattern = indices.join(',');
  const jobOverrides = [{
    job_id: 'log-entry-rate',
    analysis_config: {
      bucket_span: `${_log_analysis.bucketSpan}ms`
    },
    data_description: {
      time_field: timestampField
    },
    custom_settings: {
      logs_source_config: {
        indexPattern: indexNamePattern,
        timestampField,
        bucketSpan: _log_analysis.bucketSpan
      }
    }
  }];
  const datafeedOverrides = [{
    job_id: 'log-entry-rate',
    runtime_mappings: runtimeMappings
  }];
  const query = datasetFilter.type === 'includeSome' ? {
    bool: {
      filter: [{
        terms: {
          'event.dataset': datasetFilter.datasets
        }
      }]
    }
  } : undefined;
  return (0, _ml_setup_module_api.callSetupMlModuleAPI)({
    moduleId,
    start,
    end,
    spaceId,
    sourceId,
    indexPattern: indexNamePattern,
    jobOverrides,
    datafeedOverrides,
    query,
    useDedicatedIndex: true
  }, fetch);
};
const cleanUpModule = async (spaceId, logViewId, fetch) => {
  return await (0, _log_analysis_cleanup.cleanUpJobsAndDatafeeds)(spaceId, logViewId, _log_analysis.logEntryRateJobTypes, fetch);
};
const validateSetupIndices = async (indices, timestampField, runtimeMappings, fetch) => {
  return await (0, _validate_indices.callValidateIndicesAPI)({
    indices,
    fields: [{
      name: timestampField,
      validTypes: ['date']
    }, {
      name: _log_analysis.partitionField,
      validTypes: ['keyword']
    }],
    runtimeMappings
  }, fetch);
};
const validateSetupDatasets = async (indices, timestampField, startTime, endTime, runtimeMappings, fetch) => {
  return await (0, _validate_datasets.callValidateDatasetsAPI)({
    indices,
    timestampField,
    startTime,
    endTime,
    runtimeMappings
  }, fetch);
};
const logEntryRateModule = {
  moduleId,
  moduleName,
  moduleDescription,
  jobTypes: _log_analysis.logEntryRateJobTypes,
  bucketSpan: _log_analysis.bucketSpan,
  getJobIds,
  getJobSummary,
  getModuleDefinition,
  setUpModule,
  cleanUpModule,
  validateSetupDatasets,
  validateSetupIndices
};
exports.logEntryRateModule = logEntryRateModule;