"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateCATrustedFingerPrint = validateCATrustedFingerPrint;
exports.validateESHosts = validateESHosts;
exports.validateLogstashHosts = validateLogstashHosts;
exports.validateName = validateName;
exports.validateSSLCertificate = validateSSLCertificate;
exports.validateSSLKey = validateSSLKey;
exports.validateYamlConfig = validateYamlConfig;
var _i18n = require("@kbn/i18n");
var _jsYaml = require("js-yaml");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function validateESHosts(value) {
  const res = [];
  const urlIndexes = {};
  value.forEach((val, idx) => {
    try {
      if (!val) {
        throw new Error('Host URL required');
      }
      const urlParsed = new URL(val);
      if (!['http:', 'https:'].includes(urlParsed.protocol)) {
        throw new Error('Invalid protocol');
      }
    } catch (error) {
      res.push({
        message: _i18n.i18n.translate('xpack.fleet.settings.outputForm.elasticHostError', {
          defaultMessage: 'Invalid URL'
        }),
        index: idx
      });
    }
    const curIndexes = urlIndexes[val] || [];
    urlIndexes[val] = [...curIndexes, idx];
  });
  Object.values(urlIndexes).filter(({
    length
  }) => length > 1).forEach(indexes => {
    indexes.forEach(index => res.push({
      message: _i18n.i18n.translate('xpack.fleet.settings.outputForm.elasticHostDuplicateError', {
        defaultMessage: 'Duplicate URL'
      }),
      index
    }));
  });
  if (value.length === 0) {
    res.push({
      message: _i18n.i18n.translate('xpack.fleet.settings.outputForm.elasticUrlRequiredError', {
        defaultMessage: 'URL is required'
      })
    });
  }
  if (res.length) {
    return res;
  }
}
function validateLogstashHosts(value) {
  const res = [];
  const urlIndexes = {};
  value.forEach((val, idx) => {
    try {
      if (val.match(/^http([s]){0,1}:\/\//)) {
        res.push({
          message: _i18n.i18n.translate('xpack.fleet.settings.outputForm.logstashHostProtocolError', {
            defaultMessage: 'Host address must begin with a domain name or IP address'
          }),
          index: idx
        });
        return;
      }
      const url = new URL(`http://${val}`);
      if (url.host !== val) {
        throw new Error('Invalid host');
      }
    } catch (error) {
      if (val.length === 0) {
        res.push({
          message: _i18n.i18n.translate('xpack.fleet.settings.outputForm.logstashHostRequiredError', {
            defaultMessage: 'Host is required'
          }),
          index: idx
        });
      } else {
        res.push({
          message: _i18n.i18n.translate('xpack.fleet.settings.outputForm.logstashHostError', {
            defaultMessage: 'Invalid Host'
          }),
          index: idx
        });
      }
    }
    const curIndexes = urlIndexes[val] || [];
    urlIndexes[val] = [...curIndexes, idx];
  });
  Object.values(urlIndexes).filter(({
    length
  }) => length > 1).forEach(indexes => {
    indexes.forEach(index => res.push({
      message: _i18n.i18n.translate('xpack.fleet.settings.outputForm.logstashHostDuplicateError', {
        defaultMessage: 'Duplicate Host'
      }),
      index
    }));
  });
  if (value.length === 0) {
    res.push({
      message: _i18n.i18n.translate('xpack.fleet.settings.outputForm.logstashHostRequiredError', {
        defaultMessage: 'Host is required'
      })
    });
  }
  if (res.length) {
    return res;
  }
}
function validateYamlConfig(value) {
  try {
    (0, _jsYaml.safeLoad)(value);
    return;
  } catch (error) {
    return [_i18n.i18n.translate('xpack.fleet.settings.outputForm.invalidYamlFormatErrorMessage', {
      defaultMessage: 'Invalid YAML: {reason}',
      values: {
        reason: error.message
      }
    })];
  }
}
function validateName(value) {
  if (!value || value === '') {
    return [_i18n.i18n.translate('xpack.fleet.settings.outputForm.nameIsRequiredErrorMessage', {
      defaultMessage: 'Name is required'
    })];
  }
}
function validateCATrustedFingerPrint(value) {
  if (value !== '' && !value.match(/^[a-zA-Z0-9]+$/)) {
    return [_i18n.i18n.translate('xpack.fleet.settings.outputForm.caTrusterdFingerprintInvalidErrorMessage', {
      defaultMessage: 'CA trusted fingerprint should be valid HEX encoded SHA-256 of a CA certificate'
    })];
  }
}
function validateSSLCertificate(value) {
  if (!value || value === '') {
    return [_i18n.i18n.translate('xpack.fleet.settings.outputForm.sslCertificateRequiredErrorMessage', {
      defaultMessage: 'SSL certificate is required'
    })];
  }
}
function validateSSLKey(value) {
  if (!value || value === '') {
    return [_i18n.i18n.translate('xpack.fleet.settings.outputForm.sslKeyRequiredErrorMessage', {
      defaultMessage: 'SSL key is required'
    })];
  }
}