"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ResultSettingsLogic = void 0;
var _kea = require("kea");
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
var _flash_messages = require("../../../shared/flash_messages");
var _http = require("../../../shared/http");
var _types = require("../../../shared/schema/types");
var _engine = require("../engine");
var _constants = require("./constants");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SAVE_CONFIRMATION_MESSAGE = _i18n.i18n.translate('xpack.enterpriseSearch.appSearch.engine.resultSettings.confirmSaveMessage', {
  defaultMessage: 'The changes will start immediately. Make sure your applications are ready to accept the new search results!'
});
const RESET_CONFIRMATION_MESSAGE = _i18n.i18n.translate('xpack.enterpriseSearch.appSearch.engine.resultSettings.confirmResetMessage', {
  defaultMessage: 'Are you sure you want to restore result settings defaults? This will set all fields back to raw with no limits.'
});
const ResultSettingsLogic = (0, _kea.kea)({
  path: ['enterprise_search', 'app_search', 'result_settings_logic'],
  actions: () => ({
    initializeResultFields: (serverResultFields, schema, schemaConflicts) => {
      const resultFields = (0, _utils.convertServerResultFieldsToResultFields)(serverResultFields, schema);
      return {
        resultFields,
        schema,
        schemaConflicts
      };
    },
    clearAllFields: () => true,
    resetAllFields: () => true,
    updateField: (fieldName, settings) => ({
      fieldName,
      settings
    }),
    saving: () => true,
    clearRawSizeForField: fieldName => ({
      fieldName
    }),
    clearSnippetSizeForField: fieldName => ({
      fieldName
    }),
    toggleRawForField: fieldName => ({
      fieldName
    }),
    toggleSnippetForField: fieldName => ({
      fieldName
    }),
    toggleSnippetFallbackForField: fieldName => ({
      fieldName
    }),
    updateRawSizeForField: (fieldName, size) => ({
      fieldName,
      size
    }),
    updateSnippetSizeForField: (fieldName, size) => ({
      fieldName,
      size
    }),
    initializeResultSettingsData: () => true,
    confirmResetAllFields: () => true,
    saveResultSettings: () => true
  }),
  reducers: () => ({
    dataLoading: [true, {
      initializeResultFields: () => false
    }],
    saving: [false, {
      initializeResultFields: () => false,
      saving: () => true
    }],
    resultFields: [{}, {
      initializeResultFields: (_, {
        resultFields
      }) => resultFields,
      clearAllFields: resultFields => (0, _utils.clearAllFields)(resultFields),
      resetAllFields: resultFields => (0, _utils.resetAllFields)(resultFields),
      updateField: (resultFields, {
        fieldName,
        settings
      }) => resultFields.hasOwnProperty(fieldName) ? {
        ...resultFields,
        [fieldName]: settings
      } : resultFields
    }],
    lastSavedResultFields: [{}, {
      initializeResultFields: (_, {
        resultFields
      }) => resultFields
    }],
    schema: [{}, {
      initializeResultFields: (_, {
        schema
      }) => schema
    }],
    schemaConflicts: [{}, {
      initializeResultFields: (_, {
        schemaConflicts
      }) => schemaConflicts || {}
    }]
  }),
  selectors: ({
    selectors
  }) => ({
    validResultFields: [() => [selectors.resultFields, selectors.schema], (resultFields, schema) => Object.entries(resultFields).reduce((validResultFields, [fieldName, fieldSettings]) => {
      if (!schema[fieldName] || schema[fieldName].type === _types.SchemaType.Nested) {
        return validResultFields;
      }
      return {
        ...validResultFields,
        [fieldName]: fieldSettings
      };
    }, {})],
    textResultFields: [() => [selectors.validResultFields, selectors.schema], (resultFields, schema) => {
      const {
        textResultFields
      } = (0, _utils.splitResultFields)(resultFields, schema);
      return textResultFields;
    }],
    nonTextResultFields: [() => [selectors.validResultFields, selectors.schema], (resultFields, schema) => {
      const {
        nonTextResultFields
      } = (0, _utils.splitResultFields)(resultFields, schema);
      return nonTextResultFields;
    }],
    isSnippetAllowed: [() => [selectors.schema], schema => {
      return fieldName => {
        var _schema$fieldName;
        return !!((_schema$fieldName = schema[fieldName]) !== null && _schema$fieldName !== void 0 && _schema$fieldName.capabilities.snippet);
      };
    }],
    serverResultFields: [() => [selectors.validResultFields], resultFields => {
      return Object.entries(resultFields).reduce((serverResultFields, [fieldName, settings]) => {
        return {
          ...serverResultFields,
          [fieldName]: (0, _utils.convertToServerFieldResultSetting)(settings)
        };
      }, {});
    }],
    resultFieldsAtDefaultSettings: [() => [selectors.validResultFields], resultFields => (0, _utils.areFieldsAtDefaultSettings)(resultFields)],
    resultFieldsEmpty: [() => [selectors.validResultFields], resultFields => (0, _utils.areFieldsEmpty)(resultFields)],
    stagedUpdates: [() => [selectors.lastSavedResultFields, selectors.resultFields], (lastSavedResultFields, resultFields) => !(0, _lodash.isEqual)(lastSavedResultFields, resultFields)],
    reducedServerResultFields: [() => [selectors.serverResultFields], serverResultFields => Object.entries(serverResultFields).reduce((acc, [fieldName, resultSetting]) => {
      if (resultSetting.raw || resultSetting.snippet) {
        acc[fieldName] = resultSetting;
      }
      return acc;
    }, {})],
    queryPerformanceScore: [() => [selectors.serverResultFields, selectors.schema], (serverResultFields, schema) => {
      return Object.entries(serverResultFields).reduce((acc, [fieldName, resultField]) => {
        let newAcc = acc;
        if (resultField.raw) {
          if (schema[fieldName].type !== _types.SchemaType.Text) {
            newAcc += 0.2;
          } else if (typeof resultField.raw === 'object' && resultField.raw.size && resultField.raw.size <= 250) {
            newAcc += 1.0;
          } else {
            newAcc += 1.5;
          }
        }
        if (resultField.snippet) {
          newAcc += 2.0;
        }
        return newAcc;
      }, 0);
    }]
  }),
  listeners: ({
    actions,
    values
  }) => ({
    clearRawSizeForField: ({
      fieldName
    }) => {
      actions.updateField(fieldName, (0, _lodash.omit)(values.resultFields[fieldName], ['rawSize']));
    },
    clearSnippetSizeForField: ({
      fieldName
    }) => {
      actions.updateField(fieldName, (0, _lodash.omit)(values.resultFields[fieldName], ['snippetSize']));
    },
    toggleRawForField: ({
      fieldName
    }) => {
      // We cast this because it could be an empty object, which we can still treat as a FieldResultSetting safely
      const field = values.resultFields[fieldName];
      const raw = !field.raw;
      actions.updateField(fieldName, {
        ...(0, _lodash.omit)(field, ['rawSize']),
        raw,
        ...(raw && field.rawSize ? {
          rawSize: field.rawSize
        } : {})
      });
    },
    toggleSnippetForField: ({
      fieldName
    }) => {
      // We cast this because it could be an empty object, which we can still treat as a FieldResultSetting safely
      const field = values.resultFields[fieldName];
      const snippet = !field.snippet;
      actions.updateField(fieldName, {
        ...(0, _lodash.omit)(field, ['snippetSize']),
        snippet,
        ...(snippet ? {
          snippetSize: _constants.DEFAULT_SNIPPET_SIZE
        } : {
          snippetFallback: false
        })
      });
    },
    toggleSnippetFallbackForField: ({
      fieldName
    }) => {
      // We cast this because it could be an empty object, which we can still treat as a FieldResultSetting safely
      const field = values.resultFields[fieldName];
      actions.updateField(fieldName, {
        ...field,
        snippetFallback: !field.snippetFallback
      });
    },
    updateRawSizeForField: ({
      fieldName,
      size
    }) => {
      actions.updateField(fieldName, {
        ...values.resultFields[fieldName],
        rawSize: size
      });
    },
    updateSnippetSizeForField: ({
      fieldName,
      size
    }) => {
      actions.updateField(fieldName, {
        ...values.resultFields[fieldName],
        snippetSize: size
      });
    },
    initializeResultSettingsData: async () => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        engineName
      } = _engine.EngineLogic.values;
      const url = `/internal/app_search/engines/${engineName}/result_settings/details`;
      try {
        const {
          schema,
          schemaConflicts,
          searchSettings: {
            result_fields: serverFieldResultSettings
          }
        } = await http.get(url);
        actions.initializeResultFields(serverFieldResultSettings, schema, schemaConflicts);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    confirmResetAllFields: () => {
      if (window.confirm(RESET_CONFIRMATION_MESSAGE)) {
        actions.resetAllFields();
      }
    },
    saveResultSettings: async () => {
      if (window.confirm(SAVE_CONFIRMATION_MESSAGE)) {
        actions.saving();
        const {
          http
        } = _http.HttpLogic.values;
        const {
          engineName
        } = _engine.EngineLogic.values;
        const url = `/internal/app_search/engines/${engineName}/result_settings`;
        actions.saving();
        try {
          const response = await http.put(url, {
            body: JSON.stringify({
              result_fields: values.reducedServerResultFields
            })
          });
          actions.initializeResultFields(response.result_fields, values.schema);
          (0, _flash_messages.flashSuccessToast)(_i18n.i18n.translate('xpack.enterpriseSearch.appSearch.engine.resultSettings.saveSuccessMessage', {
            defaultMessage: 'Result settings were saved'
          }));
        } catch (e) {
          (0, _flash_messages.flashAPIErrors)(e);
        }
      }
    }
  })
});
exports.ResultSettingsLogic = ResultSettingsLogic;