"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerSavedQueryRoutes = registerSavedQueryRoutes;
var _configSchema = require("@kbn/config-schema");
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const SAVED_QUERY_ID_CONFIG = _configSchema.schema.object({
  id: _configSchema.schema.string()
});
const SAVED_QUERY_ATTRS_CONFIG = _configSchema.schema.object({
  title: _configSchema.schema.string(),
  description: _configSchema.schema.string(),
  query: _configSchema.schema.object({
    query: _configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.object({}, {
      unknowns: 'allow'
    })]),
    language: _configSchema.schema.string()
  }),
  filters: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.any())),
  timefilter: _configSchema.schema.maybe(_configSchema.schema.any())
});
const savedQueryResponseSchema = _configSchema.schema.object({
  id: _configSchema.schema.string(),
  attributes: SAVED_QUERY_ATTRS_CONFIG
});
const access = 'internal';
const version = '1';
function registerSavedQueryRoutes({
  http
}) {
  const router = http.createRouter();
  router.versioned.post({
    path: `${_constants.SAVED_QUERY_BASE_URL}/_create`,
    access
  }).addVersion({
    version,
    validate: {
      request: {
        body: SAVED_QUERY_ATTRS_CONFIG
      },
      response: {
        200: {
          body: savedQueryResponseSchema
        }
      }
    }
  }, async (context, request, response) => {
    try {
      const savedQuery = await context.savedQuery;
      const body = await savedQuery.create(request.body);
      return response.ok({
        body
      });
    } catch (e) {
      // TODO: Handle properly
      return response.customError(e);
    }
  });
  router.versioned.put({
    path: `${_constants.SAVED_QUERY_BASE_URL}/{id}`,
    access
  }).addVersion({
    version,
    validate: {
      request: {
        params: SAVED_QUERY_ID_CONFIG,
        body: SAVED_QUERY_ATTRS_CONFIG
      },
      response: {
        200: {
          body: savedQueryResponseSchema
        }
      }
    }
  }, async (context, request, response) => {
    const {
      id
    } = request.params;
    try {
      const savedQuery = await context.savedQuery;
      const body = await savedQuery.update(id, request.body);
      return response.ok({
        body
      });
    } catch (e) {
      // TODO: Handle properly
      return response.customError(e);
    }
  });
  router.versioned.get({
    path: `${_constants.SAVED_QUERY_BASE_URL}/{id}`,
    access
  }).addVersion({
    version,
    validate: {
      request: {
        params: SAVED_QUERY_ID_CONFIG
      },
      response: {
        200: {
          body: savedQueryResponseSchema
        }
      }
    }
  }, async (context, request, response) => {
    const {
      id
    } = request.params;
    try {
      const savedQuery = await context.savedQuery;
      const body = await savedQuery.get(id);
      return response.ok({
        body
      });
    } catch (e) {
      // TODO: Handle properly
      return response.customError(e);
    }
  });
  router.versioned.get({
    path: `${_constants.SAVED_QUERY_BASE_URL}/_count`,
    access
  }).addVersion({
    version,
    validate: {
      request: {},
      response: {
        200: {
          body: _configSchema.schema.number()
        }
      }
    }
  }, async (context, request, response) => {
    try {
      const savedQuery = await context.savedQuery;
      const count = await savedQuery.count();
      return response.ok({
        body: `${count}`
      });
    } catch (e) {
      // TODO: Handle properly
      return response.customError(e);
    }
  });
  router.versioned.post({
    path: `${_constants.SAVED_QUERY_BASE_URL}/_find`,
    access
  }).addVersion({
    version,
    validate: {
      request: {
        body: _configSchema.schema.object({
          search: _configSchema.schema.string({
            defaultValue: ''
          }),
          perPage: _configSchema.schema.number({
            defaultValue: 50
          }),
          page: _configSchema.schema.number({
            defaultValue: 1
          })
        })
      },
      response: {
        200: {
          body: _configSchema.schema.object({
            total: _configSchema.schema.number(),
            savedQueries: _configSchema.schema.arrayOf(savedQueryResponseSchema)
          })
        }
      }
    }
  }, async (context, request, response) => {
    try {
      const savedQuery = await context.savedQuery;
      const body = await savedQuery.find(request.body);
      return response.ok({
        body
      });
    } catch (e) {
      // TODO: Handle properly
      return response.customError(e);
    }
  });
  router.versioned.post({
    path: `${_constants.SAVED_QUERY_BASE_URL}/_all`,
    access
  }).addVersion({
    version,
    validate: {
      request: {},
      response: {
        200: {
          body: _configSchema.schema.object({
            total: _configSchema.schema.number(),
            savedQueries: _configSchema.schema.arrayOf(savedQueryResponseSchema)
          })
        }
      }
    }
  }, async (context, request, response) => {
    try {
      const savedQuery = await context.savedQuery;
      const body = await savedQuery.getAll();
      return response.ok({
        body
      });
    } catch (e) {
      // TODO: Handle properly
      return response.customError(e);
    }
  });
  router.versioned.delete({
    path: `${_constants.SAVED_QUERY_BASE_URL}/{id}`,
    access
  }).addVersion({
    version,
    validate: {
      request: {
        params: SAVED_QUERY_ID_CONFIG
      },
      response: {
        200: {
          body: _configSchema.schema.never()
        }
      }
    }
  }, async (context, request, response) => {
    const {
      id
    } = request.params;
    try {
      const savedQuery = await context.savedQuery;
      await savedQuery.delete(id);
      return response.ok();
    } catch (e) {
      // TODO: Handle properly
      return response.customError(e);
    }
  });
}