"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.filterWarnings = filterWarnings;
exports.handleWarnings = handleWarnings;
var _lodash = require("lodash");
var _eui = require("@elastic/eui");
var _public = require("@kbn/kibana-react-plugin/public");
var _react = _interopRequireDefault(require("react"));
var _services = require("../../services");
var _shard_failure_modal = require("../../shard_failure_modal");
var _extract_warnings = require("./extract_warnings");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const getDebouncedWarning = () => {
  const addWarning = () => {
    const {
      toasts
    } = (0, _services.getNotifications)();
    return (0, _lodash.debounce)(toasts.addWarning.bind(toasts), 30000, {
      leading: true
    });
  };
  const memory = {};
  return (debounceKey, title, text) => {
    memory[debounceKey] = memory[debounceKey] || addWarning();
    return memory[debounceKey]({
      title,
      text
    });
  };
};
const debouncedWarningWithoutReason = getDebouncedWarning();
const debouncedTimeoutWarning = getDebouncedWarning();
const debouncedWarning = getDebouncedWarning();

/**
 * @internal
 * All warnings are expected to come from the same response. Therefore all "text" properties, which contain the
 * response, will be the same.
 */
function handleWarnings({
  request,
  response,
  theme,
  callback,
  sessionId = '',
  requestId
}) {
  const warnings = (0, _extract_warnings.extractWarnings)(response);
  if (warnings.length === 0) {
    return;
  }
  const internal = callback ? filterWarnings(warnings, callback, request, response, requestId) : warnings;
  if (internal.length === 0) {
    return;
  }

  // timeout notification
  const [timeout] = internal.filter(w => w.type === 'timed_out');
  if (timeout) {
    debouncedTimeoutWarning(sessionId + timeout.message, timeout.message);
  }

  // shard warning failure notification
  const shardFailures = internal.filter(w => w.type === 'shard_failure');
  if (shardFailures.length === 0) {
    return;
  }
  const [warning] = shardFailures;
  const title = warning.message;

  // if warning message contains text (warning response), show in ShardFailureOpenModalButton
  if (warning.text) {
    const text = (0, _public.toMountPoint)( /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, warning.text, /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
      size: "s"
    }), /*#__PURE__*/_react.default.createElement(_eui.EuiTextAlign, {
      textAlign: "right"
    }, /*#__PURE__*/_react.default.createElement(_shard_failure_modal.ShardFailureOpenModalButton, {
      theme: theme,
      title: title,
      getRequestMeta: () => ({
        request: request,
        response
      })
    }))), {
      theme$: theme.theme$
    });
    debouncedWarning(sessionId + warning.text, title, text);
    return;
  }

  // timeout warning, or shard warning with no failure reason
  debouncedWarningWithoutReason(sessionId + title, title);
}

/**
 * @internal
 */
function filterWarnings(warnings, cb, request, response, requestId) {
  const unfiltered = [];

  // use the consumer's callback as a filter to receive warnings to handle on our side
  warnings.forEach(warning => {
    const consumerHandled = cb === null || cb === void 0 ? void 0 : cb(warning, {
      requestId,
      request,
      response
    });
    if (!consumerHandled) {
      unfiltered.push(warning);
    }
  });
  return unfiltered;
}