"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.UiSettingsService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _std = require("@kbn/std");
var _ui_settings_config = require("./ui_settings_config");
var _clients = require("./clients");
var _saved_objects = require("./saved_objects");
var _routes = require("./routes");
var _settings = require("./settings");
var _ui_settings_defaults_client = require("./clients/ui_settings_defaults_client");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/** @internal */
class UiSettingsService {
  constructor(coreContext) {
    (0, _defineProperty2.default)(this, "log", void 0);
    (0, _defineProperty2.default)(this, "config$", void 0);
    (0, _defineProperty2.default)(this, "isDist", void 0);
    (0, _defineProperty2.default)(this, "uiSettingsDefaults", new Map());
    (0, _defineProperty2.default)(this, "uiSettingsGlobalDefaults", new Map());
    (0, _defineProperty2.default)(this, "overrides", {});
    (0, _defineProperty2.default)(this, "register", (settings = {}) => {
      Object.entries(settings).forEach(([key, value]) => {
        if (this.uiSettingsDefaults.has(key)) {
          throw new Error(`uiSettings for the key [${key}] has been already registered`);
        }
        this.uiSettingsDefaults.set(key, value);
      });
    });
    (0, _defineProperty2.default)(this, "registerGlobal", (settings = {}) => {
      Object.entries(settings).forEach(([key, value]) => {
        if (this.uiSettingsGlobalDefaults.has(key)) {
          throw new Error(`Global uiSettings for the key [${key}] has been already registered`);
        }
        this.uiSettingsGlobalDefaults.set(key, value);
      });
    });
    this.coreContext = coreContext;
    this.log = coreContext.logger.get('ui-settings-service');
    this.isDist = coreContext.env.packageInfo.dist;
    this.config$ = coreContext.configService.atPath(_ui_settings_config.uiSettingsConfig.path);
  }
  async preboot() {
    this.log.debug('Prebooting ui settings service');
    const {
      overrides
    } = await (0, _rxjs.firstValueFrom)(this.config$);
    this.overrides = overrides;
    this.register((0, _settings.getCoreSettings)({
      isDist: this.isDist
    }));
    return {
      createDefaultsClient: () => new _ui_settings_defaults_client.UiSettingsDefaultsClient({
        defaults: (0, _std.mapToObject)(this.uiSettingsDefaults),
        overrides: this.overrides,
        log: this.log.get('core defaults')
      })
    };
  }
  async setup({
    http,
    savedObjects
  }) {
    this.log.debug('Setting up ui settings service');
    savedObjects.registerType(_saved_objects.uiSettingsType);
    savedObjects.registerType(_saved_objects.uiSettingsGlobalType);
    (0, _routes.registerRoutes)(http.createRouter(''));
    const config = await (0, _rxjs.firstValueFrom)(this.config$);
    this.overrides = config.overrides;
    return {
      register: this.register,
      registerGlobal: this.registerGlobal
    };
  }
  async start() {
    this.validatesDefinitions();
    this.validatesOverrides();
    return {
      asScopedToClient: this.getScopedClientFactory('namespace'),
      globalAsScopedToClient: this.getScopedClientFactory('global')
    };
  }
  async stop() {}
  getScopedClientFactory(scope) {
    const {
      version,
      buildNum
    } = this.coreContext.env.packageInfo;
    return savedObjectsClient => {
      const isNamespaceScope = scope === 'namespace';
      const options = {
        type: isNamespaceScope ? 'config' : 'config-global',
        id: (0, _std.stripVersionQualifier)(version),
        buildNum,
        savedObjectsClient,
        defaults: isNamespaceScope ? (0, _std.mapToObject)(this.uiSettingsDefaults) : (0, _std.mapToObject)(this.uiSettingsGlobalDefaults),
        overrides: isNamespaceScope ? this.overrides : {},
        log: this.log
      };
      return _clients.UiSettingsClientFactory.create(options);
    };
  }
  validatesDefinitions() {
    for (const [key, definition] of this.uiSettingsDefaults) {
      if (!definition.schema) {
        throw new Error(`Validation schema is not provided for [${key}] UI Setting`);
      }
      definition.schema.validate(definition.value, {}, `ui settings defaults [${key}]`);
    }
    for (const [key, definition] of this.uiSettingsGlobalDefaults) {
      if (!definition.schema) {
        throw new Error(`Validation schema is not provided for [${key}] Global UI Setting`);
      }
      definition.schema.validate(definition.value, {});
    }
  }
  validatesOverrides() {
    for (const [key, value] of Object.entries(this.overrides)) {
      const definition = this.uiSettingsDefaults.get(key);
      // overrides might contain UiSettings for a disabled plugin
      if (definition !== null && definition !== void 0 && definition.schema) {
        definition.schema.validate(value, {}, `ui settings overrides [${key}]`);
      }
    }
  }
}
exports.UiSettingsService = UiSettingsService;