"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.KibanaMigrator = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _coreSavedObjectsBaseServerInternal = require("@kbn/core-saved-objects-base-server-internal");
var _core = require("./core");
var _document_migrator = require("./document_migrator");
var _zdt = require("./zdt");
var _kibana_migrator_constants = require("./kibana_migrator_constants");
var _run_v2_migration = require("./run_v2_migration");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/*
 * This file contains the logic for managing the Kibana index version
 * (the shape of the mappings and documents in the index).
 */

/**
 * Manages the shape of mappings and documents in the Kibana index.
 */
class KibanaMigrator {
  /**
   * Creates an instance of KibanaMigrator.
   */
  constructor({
    client,
    typeRegistry,
    kibanaIndex,
    defaultIndexTypesMap,
    soMigrationsConfig,
    kibanaVersion,
    logger,
    docLinks,
    waitForMigrationCompletion,
    nodeRoles
  }) {
    (0, _defineProperty2.default)(this, "client", void 0);
    (0, _defineProperty2.default)(this, "documentMigrator", void 0);
    (0, _defineProperty2.default)(this, "kibanaIndex", void 0);
    (0, _defineProperty2.default)(this, "log", void 0);
    (0, _defineProperty2.default)(this, "mappingProperties", void 0);
    (0, _defineProperty2.default)(this, "typeRegistry", void 0);
    (0, _defineProperty2.default)(this, "defaultIndexTypesMap", void 0);
    (0, _defineProperty2.default)(this, "serializer", void 0);
    (0, _defineProperty2.default)(this, "migrationResult", void 0);
    (0, _defineProperty2.default)(this, "status$", new _rxjs.BehaviorSubject({
      status: 'waiting_to_start'
    }));
    (0, _defineProperty2.default)(this, "activeMappings", void 0);
    (0, _defineProperty2.default)(this, "soMigrationsConfig", void 0);
    (0, _defineProperty2.default)(this, "docLinks", void 0);
    (0, _defineProperty2.default)(this, "waitForMigrationCompletion", void 0);
    (0, _defineProperty2.default)(this, "nodeRoles", void 0);
    (0, _defineProperty2.default)(this, "kibanaVersion", void 0);
    this.client = client;
    this.kibanaIndex = kibanaIndex;
    this.soMigrationsConfig = soMigrationsConfig;
    this.typeRegistry = typeRegistry;
    this.defaultIndexTypesMap = defaultIndexTypesMap;
    this.serializer = new _coreSavedObjectsBaseServerInternal.SavedObjectsSerializer(this.typeRegistry);
    this.mappingProperties = (0, _core.buildTypesMappings)(this.typeRegistry.getAllTypes());
    this.log = logger;
    this.kibanaVersion = kibanaVersion;
    this.documentMigrator = new _document_migrator.DocumentMigrator({
      kibanaVersion: this.kibanaVersion,
      convertVersion: _kibana_migrator_constants.ALLOWED_CONVERT_VERSION,
      typeRegistry,
      log: this.log
    });
    this.waitForMigrationCompletion = waitForMigrationCompletion;
    this.nodeRoles = nodeRoles;
    // Building the active mappings (and associated md5sums) is an expensive
    // operation so we cache the result
    this.activeMappings = (0, _core.buildActiveMappings)(this.mappingProperties);
    this.docLinks = docLinks;
  }
  runMigrations({
    rerun = false
  } = {}) {
    if (this.migrationResult === undefined || rerun) {
      // Reruns are only used by CI / EsArchiver. Publishing status updates on reruns results in slowing down CI
      // unnecessarily, so we skip it in this case.
      if (!rerun) {
        this.status$.next({
          status: 'running'
        });
      }
      this.migrationResult = this.runMigrationsInternal().then(result => {
        // Similar to above, don't publish status updates when rerunning in CI.
        if (!rerun) {
          this.status$.next({
            status: 'completed',
            result
          });
        }
        return result;
      });
    }
    return this.migrationResult;
  }
  prepareMigrations() {
    this.documentMigrator.prepareMigrations();
  }
  getStatus$() {
    return this.status$.asObservable();
  }
  runMigrationsInternal() {
    const migrationAlgorithm = this.soMigrationsConfig.algorithm;
    if (migrationAlgorithm === 'zdt') {
      return (0, _zdt.runZeroDowntimeMigration)({
        kibanaVersion: this.kibanaVersion,
        kibanaIndexPrefix: this.kibanaIndex,
        typeRegistry: this.typeRegistry,
        logger: this.log,
        documentMigrator: this.documentMigrator,
        migrationConfig: this.soMigrationsConfig,
        docLinks: this.docLinks,
        serializer: this.serializer,
        elasticsearchClient: this.client,
        nodeRoles: this.nodeRoles
      });
    } else {
      return (0, _run_v2_migration.runV2Migration)({
        kibanaVersion: this.kibanaVersion,
        kibanaIndexPrefix: this.kibanaIndex,
        typeRegistry: this.typeRegistry,
        defaultIndexTypesMap: this.defaultIndexTypesMap,
        logger: this.log,
        documentMigrator: this.documentMigrator,
        migrationConfig: this.soMigrationsConfig,
        docLinks: this.docLinks,
        serializer: this.serializer,
        elasticsearchClient: this.client,
        mappingProperties: this.mappingProperties,
        waitForMigrationCompletion: this.waitForMigrationCompletion
      });
    }
  }
  getActiveMappings() {
    return this.activeMappings;
  }
  migrateDocument(doc, {
    allowDowngrade = false
  } = {}) {
    return this.documentMigrator.migrate(doc, {
      allowDowngrade
    });
  }
}
exports.KibanaMigrator = KibanaMigrator;