"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.setupOptionsListSuggestionsRoute = void 0;
var _server = require("@kbn/kibana-utils-plugin/server");
var _configSchema = require("@kbn/config-schema");
var _options_list_validation_queries = require("./options_list_validation_queries");
var _options_list_expensive_suggestion_queries = require("./options_list_expensive_suggestion_queries");
var _options_list_cheap_suggestion_queries = require("./options_list_cheap_suggestion_queries");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const setupOptionsListSuggestionsRoute = ({
  http
}, getAutocompleteSettings) => {
  const router = http.createRouter();
  router.versioned.post({
    access: 'internal',
    path: '/internal/controls/optionsList/{index}'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _configSchema.schema.object({
          index: _configSchema.schema.string()
        }, {
          unknowns: 'allow'
        }),
        body: _configSchema.schema.object({
          size: _configSchema.schema.number(),
          fieldName: _configSchema.schema.string(),
          sort: _configSchema.schema.maybe(_configSchema.schema.any()),
          filters: _configSchema.schema.maybe(_configSchema.schema.any()),
          fieldSpec: _configSchema.schema.maybe(_configSchema.schema.any()),
          allowExpensiveQueries: _configSchema.schema.boolean(),
          searchString: _configSchema.schema.maybe(_configSchema.schema.string()),
          selectedOptions: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string()))
        }, {
          unknowns: 'allow'
        })
      }
    }
  }, async (context, request, response) => {
    try {
      const suggestionRequest = request.body;
      const {
        index
      } = request.params;
      const esClient = (await context.core).elasticsearch.client.asCurrentUser;
      const suggestionsResponse = await getOptionsListSuggestions({
        abortedEvent$: request.events.aborted$,
        request: suggestionRequest,
        esClient,
        index
      });
      return response.ok({
        body: suggestionsResponse
      });
    } catch (e) {
      const kbnErr = (0, _server.getKbnServerError)(e);
      return (0, _server.reportServerError)(response, kbnErr);
    }
  });
  const getOptionsListSuggestions = async ({
    abortedEvent$,
    esClient,
    request,
    index
  }) => {
    var _suggestionBuilder$bu;
    const abortController = new AbortController();
    abortedEvent$.subscribe(() => abortController.abort());

    /**
     * Build ES Query
     */
    const {
      runPastTimeout,
      filters,
      runtimeFieldMap,
      allowExpensiveQueries
    } = request;
    const {
      terminateAfter,
      timeout
    } = getAutocompleteSettings();
    const timeoutSettings = runPastTimeout ? {} : {
      timeout: `${timeout}ms`,
      terminate_after: terminateAfter
    };
    let suggestionBuilder;
    if (allowExpensiveQueries) {
      suggestionBuilder = (0, _options_list_expensive_suggestion_queries.getExpensiveSuggestionAggregationBuilder)(request);
    } else {
      suggestionBuilder = (0, _options_list_cheap_suggestion_queries.getCheapSuggestionAggregationBuilder)(request);
    }
    const validationBuilder = (0, _options_list_validation_queries.getValidationAggregationBuilder)();
    const suggestionAggregation = (_suggestionBuilder$bu = suggestionBuilder.buildAggregation(request)) !== null && _suggestionBuilder$bu !== void 0 ? _suggestionBuilder$bu : {};
    const builtValidationAggregation = validationBuilder.buildAggregation(request);
    const validationAggregations = builtValidationAggregation ? {
      validation: builtValidationAggregation
    } : {};
    const body = {
      size: 0,
      ...timeoutSettings,
      query: {
        bool: {
          filter: filters
        }
      },
      aggs: {
        ...suggestionAggregation,
        ...validationAggregations
      },
      runtime_mappings: {
        ...runtimeFieldMap
      }
    };
    /**
     * Run ES query
     */
    const rawEsResult = await esClient.search({
      index,
      body
    }, {
      signal: abortController.signal
    });

    /**
     * Parse ES response into Options List Response
     */
    const results = suggestionBuilder.parse(rawEsResult, request);
    const totalCardinality = results.totalCardinality;
    const invalidSelections = validationBuilder.parse(rawEsResult);
    return {
      suggestions: results.suggestions,
      totalCardinality,
      invalidSelections
    };
  };
};
exports.setupOptionsListSuggestionsRoute = setupOptionsListSuggestionsRoute;