"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ContentCrud = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

class ContentCrud {
  constructor(contentTypeId, contentStorage, {
    eventBus
  }) {
    (0, _defineProperty2.default)(this, "storage", void 0);
    (0, _defineProperty2.default)(this, "eventBus", void 0);
    (0, _defineProperty2.default)(this, "contentTypeId", void 0);
    this.contentTypeId = contentTypeId;
    this.storage = contentStorage;
    this.eventBus = eventBus;
  }
  async get(ctx, contentId, options) {
    this.eventBus.emit({
      type: 'getItemStart',
      contentId,
      contentTypeId: this.contentTypeId,
      options
    });
    try {
      const item = await this.storage.get(ctx, contentId, options);
      this.eventBus.emit({
        type: 'getItemSuccess',
        contentId,
        contentTypeId: this.contentTypeId,
        data: item,
        options
      });
      return {
        contentTypeId: this.contentTypeId,
        result: item
      };
    } catch (e) {
      this.eventBus.emit({
        type: 'getItemError',
        contentId,
        contentTypeId: this.contentTypeId,
        options,
        error: e.message
      });
      throw e;
    }
  }
  async bulkGet(ctx, ids, options) {
    this.eventBus.emit({
      type: 'bulkGetItemStart',
      contentTypeId: this.contentTypeId,
      ids,
      options
    });
    try {
      const items = await this.storage.bulkGet(ctx, ids, options);
      this.eventBus.emit({
        type: 'bulkGetItemSuccess',
        ids,
        contentTypeId: this.contentTypeId,
        data: items,
        options
      });
      return {
        contentTypeId: this.contentTypeId,
        result: items
      };
    } catch (e) {
      this.eventBus.emit({
        type: 'bulkGetItemError',
        ids,
        contentTypeId: this.contentTypeId,
        options,
        error: e.message
      });
      throw e;
    }
  }
  async create(ctx, data, options) {
    this.eventBus.emit({
      type: 'createItemStart',
      contentTypeId: this.contentTypeId,
      data,
      options
    });
    try {
      const result = await this.storage.create(ctx, data, options);
      this.eventBus.emit({
        type: 'createItemSuccess',
        contentTypeId: this.contentTypeId,
        data: result,
        options
      });
      return {
        contentTypeId: this.contentTypeId,
        result
      };
    } catch (e) {
      this.eventBus.emit({
        type: 'createItemError',
        contentTypeId: this.contentTypeId,
        data,
        options,
        error: e.message
      });
      throw e;
    }
  }
  async update(ctx, id, data, options) {
    this.eventBus.emit({
      type: 'updateItemStart',
      contentId: id,
      contentTypeId: this.contentTypeId,
      data,
      options
    });
    try {
      const result = await this.storage.update(ctx, id, data, options);
      this.eventBus.emit({
        type: 'updateItemSuccess',
        contentId: id,
        contentTypeId: this.contentTypeId,
        data: result,
        options
      });
      return {
        contentTypeId: this.contentTypeId,
        result
      };
    } catch (e) {
      this.eventBus.emit({
        type: 'updateItemError',
        contentId: id,
        contentTypeId: this.contentTypeId,
        data,
        options,
        error: e.message
      });
      throw e;
    }
  }
  async delete(ctx, id, options) {
    this.eventBus.emit({
      type: 'deleteItemStart',
      contentId: id,
      contentTypeId: this.contentTypeId,
      options
    });
    try {
      const result = await this.storage.delete(ctx, id, options);
      this.eventBus.emit({
        type: 'deleteItemSuccess',
        contentId: id,
        contentTypeId: this.contentTypeId,
        options
      });
      return {
        contentTypeId: this.contentTypeId,
        result
      };
    } catch (e) {
      this.eventBus.emit({
        type: 'deleteItemError',
        contentId: id,
        contentTypeId: this.contentTypeId,
        options,
        error: e.message
      });
      throw e;
    }
  }
  async search(ctx, query, options) {
    this.eventBus.emit({
      type: 'searchItemStart',
      contentTypeId: this.contentTypeId,
      query,
      options
    });
    try {
      const result = await this.storage.search(ctx, query, options);
      this.eventBus.emit({
        type: 'searchItemSuccess',
        contentTypeId: this.contentTypeId,
        query,
        data: result,
        options
      });
      return {
        contentTypeId: this.contentTypeId,
        result
      };
    } catch (e) {
      this.eventBus.emit({
        type: 'searchItemError',
        contentTypeId: this.contentTypeId,
        query,
        options,
        error: e.message
      });
      throw e;
    }
  }
}
exports.ContentCrud = ContentCrud;