"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useLatestVulnerabilitiesByResource = exports.getQuery = void 0;
var _reactQuery = require("@tanstack/react-query");
var _rxjs = require("rxjs");
var _constants = require("../../../../common/constants");
var _get_safe_vulnerabilities_query_filter = require("../../../../common/utils/get_safe_vulnerabilities_query_filter");
var _constants2 = require("../../../common/constants");
var _use_kibana = require("../../../common/hooks/use_kibana");
var _show_error_toast = require("../../../common/utils/show_error_toast");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getQuery = ({
  query,
  sortOrder = 'desc',
  pageIndex,
  pageSize
}) => ({
  index: _constants.LATEST_VULNERABILITIES_INDEX_PATTERN,
  query: (0, _get_safe_vulnerabilities_query_filter.getSafeVulnerabilitiesQueryFilter)(query),
  aggs: {
    total: {
      cardinality: {
        field: 'resource.id'
      }
    },
    resources: {
      terms: {
        field: 'resource.id',
        size: _constants2.MAX_FINDINGS_TO_LOAD * 3,
        // in case there are more resources then size, ensuring resources with more vulnerabilities
        // will be included first, and then vulnerabilities with critical and high severity
        order: [{
          _count: sortOrder
        }, {
          critical: 'desc'
        }, {
          high: 'desc'
        }, {
          medium: 'desc'
        }]
      },
      aggs: {
        vulnerabilitiesCountBucketSort: {
          bucket_sort: {
            sort: [{
              _count: {
                order: sortOrder
              }
            }],
            from: pageIndex * pageSize,
            size: pageSize
          }
        },
        name: {
          terms: {
            field: 'resource.name',
            size: 1
          }
        },
        region: {
          terms: {
            field: 'cloud.region',
            size: 1
          }
        },
        critical: {
          filter: {
            term: {
              'vulnerability.severity': {
                value: _constants.VULNERABILITIES_SEVERITY.CRITICAL
              }
            }
          }
        },
        high: {
          filter: {
            term: {
              'vulnerability.severity': {
                value: _constants.VULNERABILITIES_SEVERITY.HIGH
              }
            }
          }
        },
        medium: {
          filter: {
            term: {
              'vulnerability.severity': {
                value: _constants.VULNERABILITIES_SEVERITY.MEDIUM
              }
            }
          }
        },
        low: {
          filter: {
            term: {
              'vulnerability.severity': {
                value: _constants.VULNERABILITIES_SEVERITY.LOW
              }
            }
          }
        }
      }
    }
  },
  size: 0
});
exports.getQuery = getQuery;
const getFirstKey = buckets => {
  return !!Array.isArray(buckets) && !!buckets.length ? buckets[0].key : '';
};
const createVulnerabilitiesByResource = resource => ({
  resource: {
    id: resource.key,
    name: getFirstKey(resource.name.buckets)
  },
  cloud: {
    region: getFirstKey(resource.region.buckets)
  },
  vulnerabilities_count: resource.doc_count,
  severity_map: {
    critical: resource.critical.doc_count,
    high: resource.high.doc_count,
    medium: resource.medium.doc_count,
    low: resource.low.doc_count
  }
});
const useLatestVulnerabilitiesByResource = options => {
  const {
    data,
    notifications: {
      toasts
    }
  } = (0, _use_kibana.useKibana)().services;
  return (0, _reactQuery.useQuery)([_constants.LATEST_VULNERABILITIES_INDEX_PATTERN, 'resource', options], async () => {
    const {
      rawResponse: {
        hits,
        aggregations
      }
    } = await (0, _rxjs.lastValueFrom)(data.search.search({
      params: getQuery(options)
    }));
    if (!aggregations) throw new Error('Failed to aggregate by resource');
    if (!Array.isArray(aggregations.resources.buckets)) throw new Error('Failed to group by, missing resource id');
    return {
      page: aggregations.resources.buckets.map(createVulnerabilitiesByResource),
      total: aggregations.total.value,
      total_vulnerabilities: hits.total
    };
  }, {
    staleTime: 5000,
    keepPreviousData: true,
    enabled: options.enabled,
    onError: err => (0, _show_error_toast.showErrorToast)(toasts, err)
  });
};
exports.useLatestVulnerabilitiesByResource = useLatestVulnerabilitiesByResource;