"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.stringifyToURL = exports.removeItemFromSessionStorage = exports.parseURL = exports.isEmptyValue = exports.isDeprecatedConnector = exports.getConnectorsFormValidators = exports.getConnectorsFormSerializer = exports.getConnectorsFormDeserializer = exports.getConnectorIcon = exports.getConnectorById = exports.convertEmptyValuesToNull = exports.connectorDeprecationValidator = void 0;
var _api = require("../../common/api");
var _validator = require("./connectors/swimlane/validator");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getConnectorById = (id, connectors) => {
  var _connectors$find;
  return (_connectors$find = connectors.find(c => c.id === id)) !== null && _connectors$find !== void 0 ? _connectors$find : null;
};
exports.getConnectorById = getConnectorById;
const validators = {
  [_api.ConnectorTypes.swimlane]: _validator.connectorValidator
};
const connectorDeprecationValidator = connector => {
  if (connector.isDeprecated) {
    return {
      message: 'Deprecated connector'
    };
  }
};
exports.connectorDeprecationValidator = connectorDeprecationValidator;
const getConnectorsFormValidators = ({
  connectors = [],
  config = {}
}) => ({
  ...config,
  validations: [{
    validator: ({
      value: connectorId
    }) => {
      const connector = getConnectorById(connectorId, connectors);
      if (connector != null) {
        return connectorDeprecationValidator(connector);
      }
    }
  }, {
    validator: ({
      value: connectorId
    }) => {
      const connector = getConnectorById(connectorId, connectors);
      if (connector != null) {
        var _validators$connector;
        return (_validators$connector = validators[connector.actionTypeId]) === null || _validators$connector === void 0 ? void 0 : _validators$connector.call(validators, connector);
      }
    }
  }]
});

/**
 * Fields without a value need to be transformed to null.
 * Passing undefined for a field to the backed will throw an error.
 * Fo that reason, we need to convert empty fields to null.
 */
exports.getConnectorsFormValidators = getConnectorsFormValidators;
const getConnectorsFormSerializer = data => {
  if (data.fields) {
    const serializedFields = convertEmptyValuesToNull(data.fields);
    return {
      ...data,
      fields: serializedFields
    };
  }
  return data;
};
exports.getConnectorsFormSerializer = getConnectorsFormSerializer;
const convertEmptyValuesToNull = fields => {
  if (fields) {
    return Object.entries(fields).reduce((acc, [key, value]) => {
      return {
        ...acc,
        [key]: isEmptyValue(value) ? null : value
      };
    }, {});
  }
  return null;
};

/**
 * We cannot use lodash isEmpty util function
 * because it will return true for primitive values
 * like boolean or numbers
 */
exports.convertEmptyValuesToNull = convertEmptyValuesToNull;
const isEmptyValue = value => value === null || value === undefined || typeof value === 'object' && Object.keys(value).length === 0 || typeof value === 'string' && value.trim().length === 0;

/**
 * Form html elements do not support null values.
 * For that reason, we need to convert null values to
 * undefined which is supported.
 */
exports.isEmptyValue = isEmptyValue;
const getConnectorsFormDeserializer = data => {
  if (data.fields) {
    const deserializedFields = Object.entries(data.fields).reduce((acc, [key, value]) => ({
      ...acc,
      [key]: value === null ? undefined : value
    }), {});
    return {
      ...data,
      fields: deserializedFields
    };
  }
  return data;
};
exports.getConnectorsFormDeserializer = getConnectorsFormDeserializer;
const getConnectorIcon = (triggersActionsUi, type) => {
  /**
   * triggersActionsUi.actionTypeRegistry.get will throw an error if the type is not registered.
   * This will break Kibana if not handled properly.
   */
  const emptyResponse = '';
  if (type == null) {
    return emptyResponse;
  }
  try {
    if (triggersActionsUi.actionTypeRegistry.has(type)) {
      return triggersActionsUi.actionTypeRegistry.get(type).iconClass;
    }
  } catch {
    return emptyResponse;
  }
  return emptyResponse;
};
exports.getConnectorIcon = getConnectorIcon;
const isDeprecatedConnector = connector => {
  var _connector$isDeprecat;
  return (_connector$isDeprecat = connector === null || connector === void 0 ? void 0 : connector.isDeprecated) !== null && _connector$isDeprecat !== void 0 ? _connector$isDeprecat : false;
};
exports.isDeprecatedConnector = isDeprecatedConnector;
const removeItemFromSessionStorage = key => {
  window.sessionStorage.removeItem(key);
};
exports.removeItemFromSessionStorage = removeItemFromSessionStorage;
const stringifyToURL = parsedParams => new URLSearchParams(parsedParams).toString();
exports.stringifyToURL = stringifyToURL;
const parseURL = queryString => Object.fromEntries(new URLSearchParams(queryString));
exports.parseURL = parseURL;