"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createInternalESClient = createInternalESClient;
exports.createInternalESClientWithContext = createInternalESClientWithContext;
var _server = require("@kbn/observability-plugin/server");
var _call_async_with_debug = require("../call_async_with_debug");
var _cancel_es_request_on_abort = require("../cancel_es_request_on_abort");
var _get_apm_indices = require("../../../../routes/settings/apm_indices/get_apm_indices");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function createInternalESClientWithContext({
  debug,
  config,
  request,
  context
}) {
  const coreContext = await context.core;
  const {
    asInternalUser
  } = coreContext.elasticsearch.client;
  const savedObjectsClient = coreContext.savedObjects.client;
  return createInternalESClient({
    debug,
    config,
    request,
    savedObjectsClient,
    elasticsearchClient: asInternalUser
  });
}
async function createInternalESClient({
  debug,
  config,
  request,
  savedObjectsClient,
  elasticsearchClient
}) {
  function callEs(operationName, {
    makeRequestWithSignal,
    requestType,
    params
  }) {
    return (0, _call_async_with_debug.callAsyncWithDebug)({
      cb: () => {
        const controller = new AbortController();
        const res = makeRequestWithSignal(controller.signal);
        return (0, _server.unwrapEsResponse)(request ? (0, _cancel_es_request_on_abort.cancelEsRequestOnAbort)(res, request, controller) : res);
      },
      getDebugMessage: () => {
        return {
          title: request ? (0, _call_async_with_debug.getDebugTitle)(request) : 'Internal request',
          body: (0, _call_async_with_debug.getDebugBody)({
            params,
            requestType,
            operationName
          })
        };
      },
      debug,
      isCalledWithInternalUser: true,
      request,
      requestParams: params,
      operationName
    });
  }
  return {
    apmIndices: await (0, _get_apm_indices.getApmIndices)({
      savedObjectsClient,
      config
    }),
    search: async (operationName, params) => {
      return callEs(operationName, {
        requestType: 'search',
        makeRequestWithSignal: signal => elasticsearchClient.search(params, {
          signal,
          meta: true
        }),
        params
      });
    },
    index: (operationName, params) => {
      return callEs(operationName, {
        requestType: 'index',
        makeRequestWithSignal: signal => elasticsearchClient.index(params, {
          signal,
          meta: true
        }),
        params
      });
    },
    delete: (operationName, params) => {
      return callEs(operationName, {
        requestType: 'delete',
        makeRequestWithSignal: signal => elasticsearchClient.delete(params, {
          signal,
          meta: true
        }),
        params
      });
    },
    indicesCreate: (operationName, params) => {
      return callEs(operationName, {
        requestType: 'indices.create',
        makeRequestWithSignal: signal => elasticsearchClient.indices.create(params, {
          signal,
          meta: true
        }),
        params
      });
    }
  };
}