"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.THRESHOLD_MET_GROUP_ID = exports.RULE_TYPES_CONFIG = exports.ApmRuleType = exports.AggregationType = exports.APM_SERVER_FEATURE_ID = exports.ANOMALY_ALERT_SEVERITY_TYPES = void 0;
exports.formatAnomalyReason = formatAnomalyReason;
exports.formatErrorCountReason = formatErrorCountReason;
exports.formatTransactionDurationReason = formatTransactionDurationReason;
exports.formatTransactionErrorRateReason = formatTransactionErrorRateReason;
exports.getFieldValueLabel = void 0;
var _i18n = require("@kbn/i18n");
var _common = require("@kbn/observability-plugin/common");
var _anomaly_severity = require("@kbn/ml-anomaly-utils/anomaly_severity");
var _anomaly_threshold = require("@kbn/ml-anomaly-utils/anomaly_threshold");
var _apm = require("../es_fields/apm");
var _environment_filter_values = require("../environment_filter_values");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const APM_SERVER_FEATURE_ID = 'apm';
exports.APM_SERVER_FEATURE_ID = APM_SERVER_FEATURE_ID;
let ApmRuleType;
exports.ApmRuleType = ApmRuleType;
(function (ApmRuleType) {
  ApmRuleType["ErrorCount"] = "apm.error_rate";
  ApmRuleType["TransactionErrorRate"] = "apm.transaction_error_rate";
  ApmRuleType["TransactionDuration"] = "apm.transaction_duration";
  ApmRuleType["Anomaly"] = "apm.anomaly";
})(ApmRuleType || (exports.ApmRuleType = ApmRuleType = {}));
let AggregationType;
exports.AggregationType = AggregationType;
(function (AggregationType) {
  AggregationType["Avg"] = "avg";
  AggregationType["P95"] = "95th";
  AggregationType["P99"] = "99th";
})(AggregationType || (exports.AggregationType = AggregationType = {}));
const THRESHOLD_MET_GROUP_ID = 'threshold_met';
exports.THRESHOLD_MET_GROUP_ID = THRESHOLD_MET_GROUP_ID;
const THRESHOLD_MET_GROUP = {
  id: THRESHOLD_MET_GROUP_ID,
  name: _i18n.i18n.translate('xpack.apm.a.thresholdMet', {
    defaultMessage: 'Threshold met'
  })
};
const getFieldNameLabel = field => {
  switch (field) {
    case _apm.SERVICE_NAME:
      return 'service';
    case _apm.SERVICE_ENVIRONMENT:
      return 'env';
    case _apm.TRANSACTION_TYPE:
      return 'type';
    case _apm.TRANSACTION_NAME:
      return 'name';
    case _apm.ERROR_GROUP_ID:
      return 'error key';
    default:
      return field;
  }
};
const getFieldValueLabel = (field, fieldValue) => {
  return field === _apm.SERVICE_ENVIRONMENT ? (0, _environment_filter_values.getEnvironmentLabel)(fieldValue) : fieldValue;
};
exports.getFieldValueLabel = getFieldValueLabel;
const formatGroupByFields = groupByFields => {
  const groupByFieldLabels = Object.keys(groupByFields).map(field => `${getFieldNameLabel(field)}: ${getFieldValueLabel(field, groupByFields[field])}`);
  return groupByFieldLabels.join(', ');
};
function formatErrorCountReason({
  threshold,
  measured,
  windowSize,
  windowUnit,
  groupByFields
}) {
  return _i18n.i18n.translate('xpack.apm.alertTypes.errorCount.reason', {
    defaultMessage: `Error count is {measured} in the last {interval} for {group}. Alert when > {threshold}.`,
    values: {
      threshold,
      measured,
      interval: (0, _common.formatDurationFromTimeUnitChar)(windowSize, windowUnit),
      group: formatGroupByFields(groupByFields)
    }
  });
}
function formatTransactionDurationReason({
  threshold,
  measured,
  asDuration,
  aggregationType,
  windowSize,
  windowUnit,
  groupByFields
}) {
  let aggregationTypeFormatted = aggregationType.charAt(0).toUpperCase() + aggregationType.slice(1);
  if (aggregationTypeFormatted === 'Avg') aggregationTypeFormatted = aggregationTypeFormatted + '.';
  return _i18n.i18n.translate('xpack.apm.alertTypes.transactionDuration.reason', {
    defaultMessage: `{aggregationType} latency is {measured} in the last {interval} for {group}. Alert when > {threshold}.`,
    values: {
      threshold: asDuration(threshold),
      measured: asDuration(measured),
      aggregationType: aggregationTypeFormatted,
      interval: (0, _common.formatDurationFromTimeUnitChar)(windowSize, windowUnit),
      group: formatGroupByFields(groupByFields)
    }
  });
}
function formatTransactionErrorRateReason({
  threshold,
  measured,
  asPercent,
  windowSize,
  windowUnit,
  groupByFields
}) {
  return _i18n.i18n.translate('xpack.apm.alertTypes.transactionErrorRate.reason', {
    defaultMessage: `Failed transactions is {measured} in the last {interval} for {group}. Alert when > {threshold}.`,
    values: {
      threshold: asPercent(threshold, 100),
      measured: asPercent(measured, 100),
      interval: (0, _common.formatDurationFromTimeUnitChar)(windowSize, windowUnit),
      group: formatGroupByFields(groupByFields)
    }
  });
}
function formatAnomalyReason({
  serviceName,
  severityLevel,
  measured,
  windowSize,
  windowUnit
}) {
  return _i18n.i18n.translate('xpack.apm.alertTypes.transactionDurationAnomaly.reason', {
    defaultMessage: `{severityLevel} anomaly with a score of {measured} was detected in the last {interval} for {serviceName}.`,
    values: {
      serviceName,
      severityLevel,
      measured,
      interval: (0, _common.formatDurationFromTimeUnitChar)(windowSize, windowUnit)
    }
  });
}
const RULE_TYPES_CONFIG = {
  [ApmRuleType.ErrorCount]: {
    name: _i18n.i18n.translate('xpack.apm.errorCountAlert.name', {
      defaultMessage: 'Error count threshold'
    }),
    actionGroups: [THRESHOLD_MET_GROUP],
    defaultActionGroupId: THRESHOLD_MET_GROUP_ID,
    minimumLicenseRequired: 'basic',
    producer: APM_SERVER_FEATURE_ID,
    isExportable: true
  },
  [ApmRuleType.TransactionDuration]: {
    name: _i18n.i18n.translate('xpack.apm.transactionDurationAlert.name', {
      defaultMessage: 'Latency threshold'
    }),
    actionGroups: [THRESHOLD_MET_GROUP],
    defaultActionGroupId: THRESHOLD_MET_GROUP_ID,
    minimumLicenseRequired: 'basic',
    producer: APM_SERVER_FEATURE_ID,
    isExportable: true
  },
  [ApmRuleType.Anomaly]: {
    name: _i18n.i18n.translate('xpack.apm.anomalyAlert.name', {
      defaultMessage: 'Anomaly'
    }),
    actionGroups: [THRESHOLD_MET_GROUP],
    defaultActionGroupId: THRESHOLD_MET_GROUP_ID,
    minimumLicenseRequired: 'basic',
    producer: APM_SERVER_FEATURE_ID,
    isExportable: true
  },
  [ApmRuleType.TransactionErrorRate]: {
    name: _i18n.i18n.translate('xpack.apm.transactionErrorRateAlert.name', {
      defaultMessage: 'Failed transaction rate threshold'
    }),
    actionGroups: [THRESHOLD_MET_GROUP],
    defaultActionGroupId: THRESHOLD_MET_GROUP_ID,
    minimumLicenseRequired: 'basic',
    producer: APM_SERVER_FEATURE_ID,
    isExportable: true
  }
};
exports.RULE_TYPES_CONFIG = RULE_TYPES_CONFIG;
const ANOMALY_ALERT_SEVERITY_TYPES = [{
  type: _anomaly_severity.ML_ANOMALY_SEVERITY.CRITICAL,
  label: _i18n.i18n.translate('xpack.apm.alerts.anomalySeverity.criticalLabel', {
    defaultMessage: 'critical'
  }),
  threshold: _anomaly_threshold.ML_ANOMALY_THRESHOLD.CRITICAL
}, {
  type: _anomaly_severity.ML_ANOMALY_SEVERITY.MAJOR,
  label: _i18n.i18n.translate('xpack.apm.alerts.anomalySeverity.majorLabel', {
    defaultMessage: 'major'
  }),
  threshold: _anomaly_threshold.ML_ANOMALY_THRESHOLD.MAJOR
}, {
  type: _anomaly_severity.ML_ANOMALY_SEVERITY.MINOR,
  label: _i18n.i18n.translate('xpack.apm.alerts.anomalySeverity.minor', {
    defaultMessage: 'minor'
  }),
  threshold: _anomaly_threshold.ML_ANOMALY_THRESHOLD.MINOR
}, {
  type: _anomaly_severity.ML_ANOMALY_SEVERITY.WARNING,
  label: _i18n.i18n.translate('xpack.apm.alerts.anomalySeverity.warningLabel', {
    defaultMessage: 'warning'
  }),
  threshold: _anomaly_threshold.ML_ANOMALY_THRESHOLD.WARNING
}];

// Server side registrations
// x-pack/plugins/apm/server/lib/alerts/<alert>.ts
// x-pack/plugins/apm/server/lib/alerts/register_apm_alerts.ts

// Client side registrations:
// x-pack/plugins/apm/public/components/alerting/<alert>/index.tsx
// x-pack/plugins/apm/public/components/alerting/register_apm_alerts
exports.ANOMALY_ALERT_SEVERITY_TYPES = ANOMALY_ALERT_SEVERITY_TYPES;