"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RulesSettingsFlappingClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _common = require("../../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const verifyFlappingSettings = flappingSettings => {
  const {
    lookBackWindow,
    statusChangeThreshold
  } = flappingSettings;
  if (lookBackWindow < _common.MIN_LOOK_BACK_WINDOW || lookBackWindow > _common.MAX_LOOK_BACK_WINDOW) {
    throw _boom.default.badRequest(`Invalid lookBackWindow value, must be between ${_common.MIN_LOOK_BACK_WINDOW} and ${_common.MAX_LOOK_BACK_WINDOW}, but got: ${lookBackWindow}.`);
  }
  if (statusChangeThreshold < _common.MIN_STATUS_CHANGE_THRESHOLD || statusChangeThreshold > _common.MAX_STATUS_CHANGE_THRESHOLD) {
    throw _boom.default.badRequest(`Invalid statusChangeThreshold value, must be between ${_common.MIN_STATUS_CHANGE_THRESHOLD} and ${_common.MAX_STATUS_CHANGE_THRESHOLD}, but got: ${statusChangeThreshold}.`);
  }
  if (lookBackWindow < statusChangeThreshold) {
    throw _boom.default.badRequest(`Invalid values,lookBackWindow (${lookBackWindow}) must be equal to or greater than statusChangeThreshold (${statusChangeThreshold}).`);
  }
};
class RulesSettingsFlappingClient {
  constructor(options) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "savedObjectsClient", void 0);
    (0, _defineProperty2.default)(this, "getOrCreate", void 0);
    (0, _defineProperty2.default)(this, "getModificationMetadata", void 0);
    this.logger = options.logger;
    this.savedObjectsClient = options.savedObjectsClient;
    this.getOrCreate = options.getOrCreate;
    this.getModificationMetadata = options.getModificationMetadata;
  }
  async get() {
    const rulesSettings = await this.getOrCreate();
    return rulesSettings.attributes.flapping;
  }
  async update(newFlappingProperties) {
    try {
      verifyFlappingSettings(newFlappingProperties);
    } catch (e) {
      this.logger.error(`Failed to verify new flapping settings properties when updating. Error: ${e}`);
      throw e;
    }
    const {
      attributes,
      version
    } = await this.getOrCreate();
    const modificationMetadata = await this.getModificationMetadata();
    try {
      const result = await this.savedObjectsClient.update(_common.RULES_SETTINGS_SAVED_OBJECT_TYPE, _common.RULES_SETTINGS_SAVED_OBJECT_ID, {
        ...attributes,
        flapping: {
          ...attributes.flapping,
          ...newFlappingProperties,
          updatedAt: modificationMetadata.updatedAt,
          updatedBy: modificationMetadata.updatedBy
        }
      }, {
        version
      });
      return result.attributes.flapping;
    } catch (e) {
      const errorMessage = 'savedObjectsClient errored trying to update flapping settings';
      this.logger.error(`${errorMessage}: ${e}`);
      throw _boom.default.boomify(e, {
        message: errorMessage
      });
    }
  }
}
exports.RulesSettingsFlappingClient = RulesSettingsFlappingClient;