"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getMaintenanceWindowDateAndStatus = exports.findRecentEventWithStatus = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _common = require("../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Returns the most recent/relevant event and the status for a maintenance window
const getMaintenanceWindowDateAndStatus = ({
  events,
  dateToCompare,
  expirationDate
}) => {
  // No events, status is finished or archived
  if (!events.length) {
    const status = _moment.default.utc(expirationDate).isBefore(dateToCompare) ? _common.MaintenanceWindowStatus.Archived : _common.MaintenanceWindowStatus.Finished;
    return {
      eventStartTime: null,
      eventEndTime: null,
      status
    };
  }
  const {
    event,
    status,
    index
  } = findRecentEventWithStatus(events, dateToCompare);
  // Past expiration, show the last event, but status is now archived
  if (_moment.default.utc(expirationDate).isBefore(dateToCompare)) {
    return {
      eventStartTime: event.gte,
      eventEndTime: event.lte,
      status: _common.MaintenanceWindowStatus.Archived,
      index
    };
  }
  return {
    eventStartTime: event.gte,
    eventEndTime: event.lte,
    status,
    index
  };
};

// Binary date search to find the closest (or running) event relative to an arbitrary date
exports.getMaintenanceWindowDateAndStatus = getMaintenanceWindowDateAndStatus;
const findRecentEventWithStatus = (events, dateToCompare) => {
  const result = binaryDateSearch(events, dateToCompare, 0, events.length - 1);
  // Has running or upcoming event, just return the event
  if (result.status === _common.MaintenanceWindowStatus.Running || result.status === _common.MaintenanceWindowStatus.Upcoming) {
    return result;
  }
  // At the last event and it's finished, no more events are schedule so just return
  if (result.status === _common.MaintenanceWindowStatus.Finished && result.index === events.length - 1) {
    return result;
  }
  return {
    event: events[result.index + 1],
    status: _common.MaintenanceWindowStatus.Upcoming,
    index: result.index + 1
  };
};

// Get the maintenance window status of any particular event relative to an arbitrary date
exports.findRecentEventWithStatus = findRecentEventWithStatus;
const getEventStatus = (event, dateToCompare) => {
  if (_moment.default.utc(event.gte).isAfter(dateToCompare)) {
    return _common.MaintenanceWindowStatus.Upcoming;
  }
  if (_moment.default.utc(event.lte).isSameOrBefore(dateToCompare)) {
    return _common.MaintenanceWindowStatus.Finished;
  }
  return _common.MaintenanceWindowStatus.Running;
};
const binaryDateSearch = (events, dateToCompare, startIndex, endIndex, lastIndex) => {
  // Base case, take the last event it checked to see what the relative status to that event is
  if (startIndex > endIndex && typeof lastIndex === 'number') {
    const event = events[lastIndex];
    if (event) {
      return {
        event,
        status: getEventStatus(event, dateToCompare),
        index: lastIndex
      };
    }
  }
  const midIndex = startIndex + Math.floor((endIndex - startIndex) / 2);
  const midEvent = events[midIndex];
  const midEventStatus = getEventStatus(midEvent, dateToCompare);
  switch (midEventStatus) {
    case _common.MaintenanceWindowStatus.Running:
      return {
        event: midEvent,
        status: _common.MaintenanceWindowStatus.Running,
        index: midIndex
      };
    case _common.MaintenanceWindowStatus.Upcoming:
      return binaryDateSearch(events, dateToCompare, startIndex, midIndex - 1, midIndex);
    case _common.MaintenanceWindowStatus.Finished:
      return binaryDateSearch(events, dateToCompare, midIndex + 1, endIndex, midIndex);
  }
};