"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useDebouncedValue = void 0;
var _react = require("react");
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Debounces value changes and updates inputValue on root state changes if no debounced changes
 * are in flight because the user is currently modifying the value.
 *
 * * allowFalsyValue: update upstream with all falsy values but null or undefined
 *
 * When testing this function mock the "debounce" function in lodash (see this module test for an example)
 */

const useDebouncedValue = ({
  onChange,
  value,
  defaultValue
}, {
  allowFalsyValue
} = {}) => {
  const [inputValue, setInputValue] = (0, _react.useState)(value);
  const unflushedChanges = (0, _react.useRef)(false);
  const shouldUpdateWithFalsyValue = Boolean(allowFalsyValue);

  // Save the initial value
  const initialValue = (0, _react.useRef)(defaultValue !== null && defaultValue !== void 0 ? defaultValue : value);
  const flushChangesTimeout = (0, _react.useRef)();
  const onChangeDebounced = (0, _react.useMemo)(() => {
    const callback = (0, _lodash.debounce)(val => {
      onChange(val);
      // do not reset unflushed flag right away, wait a bit for upstream to pick it up
      flushChangesTimeout.current = setTimeout(() => {
        unflushedChanges.current = false;
      }, 256);
    }, 256);
    return val => {
      if (flushChangesTimeout.current) {
        clearTimeout(flushChangesTimeout.current);
      }
      unflushedChanges.current = true;
      callback(val);
    };
  }, [onChange]);
  (0, _react.useEffect)(() => {
    if (!unflushedChanges.current && value !== inputValue) {
      setInputValue(value);
    }
  }, [value, inputValue]);
  const handleInputChange = val => {
    setInputValue(val);
    const valueToUpload = shouldUpdateWithFalsyValue ? val !== null && val !== void 0 ? val : initialValue.current : val || initialValue.current;
    onChangeDebounced(valueToUpload);
  };
  return {
    inputValue,
    handleInputChange,
    initialValue: initialValue.current
  };
};
exports.useDebouncedValue = useDebouncedValue;