"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Handler = void 0;
var _d = _interopRequireDefault(require("d3"));
var _lodash = _interopRequireDefault(require("lodash"));
var _markdownIt = _interopRequireDefault(require("markdown-it"));
var _moment = _interopRequireDefault(require("moment"));
var _public = require("@kbn/kibana-utils-plugin/public");
var _vis_types = require("../visualizations/vis_types");
var _errors = require("../errors");
var _layout = require("./layout/layout");
var _chart_title = require("./chart_title");
var _axis = require("./axis/axis");
var _chart_grid = require("./chart_grid");
var _binder = require("./binder");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const markdownIt = new _markdownIt.default({
  html: false,
  linkify: true
});
const convertToTimestamp = date => {
  return parseInt((0, _moment.default)(date).format('x'));
};

/**
 * Handles building all the components of the visualization
 *
 * @class Handler
 * @constructor
 * @param vis {Object} Reference to the Vis Class Constructor
 * @param opts {Object} Reference to Visualization constructors needed to
 * create the visualization
 */
class Handler {
  constructor(vis, visConfig, uiSettings) {
    this.el = visConfig.get('el');
    this.ChartClass = _vis_types.visTypes[visConfig.get('type')];
    this.uiSettings = uiSettings;
    this.charts = [];
    this.vis = vis;
    this.visConfig = visConfig;
    this.data = visConfig.data;
    this.categoryAxes = visConfig.get('categoryAxes').map(axisArgs => new _axis.Axis(visConfig, axisArgs));
    this.valueAxes = visConfig.get('valueAxes').map(axisArgs => new _axis.Axis(visConfig, axisArgs));
    this.chartTitle = new _chart_title.ChartTitle(visConfig);
    this.grid = new _chart_grid.ChartGrid(this, visConfig.get('grid'));
    if (visConfig.get('type') === 'point_series') {
      this.data.stackData(this);
    }
    if (visConfig.get('resize', false)) {
      this.resize = visConfig.get('resize');
    }
    this.layout = new _layout.Layout(visConfig);
    this.binder = new _binder.Binder();
    this.renderArray = _lodash.default.filter([this.layout, this.chartTitle], Boolean);
    this.renderArray = this.renderArray.concat(this.valueAxes)
    // category axes need to render in reverse order https://github.com/elastic/kibana/issues/13551
    .concat(this.categoryAxes.slice().reverse());

    // memoize so that the same function is returned every time,
    // allowing us to remove/re-add the same function
    this.getProxyHandler = _lodash.default.memoize(function (eventType) {
      const self = this;
      return function (eventPayload) {
        var _eventPayload$data$se;
        switch (eventType) {
          case 'brush':
            const {
              xRaw
            } = (_eventPayload$data$se = eventPayload.data.series[0]) === null || _eventPayload$data$se === void 0 ? void 0 : _eventPayload$data$se.values.find(({
              xRaw
            }) => Boolean(xRaw));
            if (!xRaw) return; // not sure if this is possible?
            const [start, end] = eventPayload.range;
            const range = [convertToTimestamp(start), convertToTimestamp(end)];
            return self.vis.emit(eventType, {
              name: 'brush',
              data: {
                table: xRaw.table,
                range,
                column: xRaw.column
              }
            });
          case 'click':
            return self.vis.emit(eventType, {
              name: 'filter',
              data: eventPayload
            });
        }
      };
    });

    /**
     * Enables events, i.e. binds specific events to the chart
     * object(s) `on` method. For example, `click` or `mousedown` events.
     *
     * @method enable
     * @param event {String} Event type
     * @param chart {Object} Chart
     * @returns {*}
     */
    this.enable = this.chartEventProxyToggle('on');

    /**
     * Disables events for all charts
     *
     * @method disable
     * @param event {String} Event type
     * @param chart {Object} Chart
     * @returns {*}
     */
    this.disable = this.chartEventProxyToggle('off');
  }
  /**
   * Validates whether data is actually present in the data object
   * used to render the Vis. Throws a no results error if data is not
   * present.
   *
   * @private
   */
  _validateData() {
    const dataType = this.data.type;
    if (!dataType) {
      throw new _errors.NoResults();
    }
  }

  /**
   * Renders the constructors that create the visualization,
   * including the chart constructor
   *
   * @method render
   * @returns {HTMLElement} With the visualization child element
   */
  render() {
    if (this.visConfig.get('error', null)) return this.error(this.visConfig.get('error'));
    const self = this;
    const {
      binder,
      charts = []
    } = this;
    const selection = _d.default.select(this.el);
    selection.selectAll('*').remove();
    this._validateData();
    this.renderArray.forEach(function (property) {
      if (typeof property.render === 'function') {
        property.render();
      }
    });

    // render the chart(s)
    let loadedCount = 0;
    const chartSelection = selection.selectAll('.chart');
    chartSelection.each(function (chartData) {
      const chart = new self.ChartClass(self, this, chartData, self.uiSettings);
      self.vis.eventNames().forEach(function (event) {
        self.enable(event, chart);
      });
      binder.on(chart.events, 'rendered', () => {
        loadedCount++;
        if (loadedCount === chartSelection.length) {
          // events from all charts are propagated to vis, we only need to fire renderComplete once they all finish
          self.vis.emit('renderComplete');
        }
      });
      charts.push(chart);
      chart.render();
    });
  }
  chartEventProxyToggle(method) {
    return function (event, chart) {
      const proxyHandler = this.getProxyHandler(event);
      _lodash.default.each(chart ? [chart] : this.charts, function (chart) {
        chart.events[method](event, proxyHandler);
      });
    };
  }

  /**
   * Removes all DOM elements from the HTML element provided
   *
   * @method removeAll
   * @param el {HTMLElement} Reference to the HTML Element that
   * contains the chart
   * @returns {D3.Selection|D3.Transition.Transition} With the chart
   * child element removed
   */
  removeAll(el) {
    return _d.default.select(el).selectAll('*').remove();
  }

  /**
   * Displays an error message in the DOM
   *
   * @method error
   * @param message {String} Error message to display
   * @returns {HTMLElement} Displays the input message
   */
  error(message) {
    this.removeAll(this.el);
    const div = _d.default.select(this.el).append('div')
    // class name needs `chart` in it for the polling checkSize function
    // to continuously call render on resize
    .attr('class', 'visError chart error').attr('data-test-subj', 'vislibVisualizeError');
    div.append('h4').text(markdownIt.renderInline(message));
    (0, _public.dispatchRenderComplete)(this.el);
    return div;
  }

  /**
   * Destroys all the charts in the visualization
   *
   * @method destroy
   */
  destroy() {
    this.binder.destroy();
    this.renderArray.forEach(function (renderable) {
      if (_lodash.default.isFunction(renderable.destroy)) {
        renderable.destroy();
      }
    });
    this.charts.splice(0).forEach(function (chart) {
      if (_lodash.default.isFunction(chart.destroy)) {
        chart.destroy();
      }
    });
  }
}
exports.Handler = Handler;