"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Axis = void 0;
var _d = _interopRequireDefault(require("d3"));
var _lodash = _interopRequireDefault(require("lodash"));
var _jquery = _interopRequireDefault(require("jquery"));
var _error_handler = require("../_error_handler");
var _axis_title = require("./axis_title");
var _axis_labels = require("./axis_labels");
var _axis_scale = require("./axis_scale");
var _axis_config = require("./axis_config");
var _errors = require("../../errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

class Axis extends _error_handler.ErrorHandler {
  constructor(visConfig, axisConfigArgs) {
    super();
    this.visConfig = visConfig;
    this.axisConfig = new _axis_config.AxisConfig(this.visConfig, axisConfigArgs);
    if (this.axisConfig.get('type') === 'category') {
      this.values = this.axisConfig.values;
      this.ordered = this.axisConfig.ordered;
    }
    this.axisScale = new _axis_scale.AxisScale(this.axisConfig, visConfig);
    this.axisTitle = new _axis_title.AxisTitle(this.axisConfig);
    this.axisLabels = new _axis_labels.AxisLabels(this.axisConfig, this.axisScale);
    this.stack = _d.default.layout.stack().x(d => {
      return d.x;
    }).y(d => {
      if (typeof this.axisConfig.get('scale.offset') === 'function' && this.axisConfig.get('scale.offset').name === 'expand') {
        return Math.abs(d.y);
      }
      return d.y;
    }).offset(this.axisConfig.get('scale.offset', 'zero'));
    const stackedMode = ['normal', 'grouped'].includes(this.axisConfig.get('scale.mode'));
    if (stackedMode) {
      this.stack = this._stackNegAndPosVals;
    }
  }
  _stackNegAndPosVals(data) {
    const cache = {};
    data.forEach(series => {
      series.forEach(value => {
        if (!cache[value.x]) cache[value.x] = [0, 0];
        value.y0 = cache[value.x][value.y < 0 ? 0 : 1];
        cache[value.x][value.y < 0 ? 0 : 1] += value.y;
      });
    });
    return data;
  }
  render() {
    const elSelector = this.axisConfig.get('elSelector');
    const rootEl = this.axisConfig.get('rootEl');
    _d.default.select(rootEl).selectAll(elSelector).call(this.draw());
  }
  destroy() {
    const elSelector = this.axisConfig.get('elSelector');
    const rootEl = this.axisConfig.get('rootEl');
    (0, _jquery.default)(rootEl).find(elSelector).find('svg').remove();
    this.axisTitle.destroy();
  }
  getAxis(length) {
    const scale = this.axisScale.getScale(length);
    const position = this.axisConfig.get('position');
    const axisFormatter = this.axisConfig.get('labels.axisFormatter');
    const d3Axis = _d.default.svg.axis().scale(scale).tickFormat(axisFormatter).orient(position);
    if (this.axisConfig.isTimeDomain()) {
      // use custom overwritten tick function on time domains to get nice
      // tick positions (e.g. at the start of the day) even for custom timezones
      d3Axis.tickValues(scale.timezoneCorrectedTicks(this.tickScale(length)));
    } else {
      d3Axis.ticks(this.tickScale(length));
    }
    return d3Axis;
  }
  getScale() {
    return this.axisScale.scale;
  }
  addInterval(interval) {
    return this.axisScale.addInterval(interval);
  }
  substractInterval(interval) {
    return this.axisScale.substractInterval(interval);
  }
  tickScale(length) {
    const yTickScale = _d.default.scale.linear().clamp(true).domain([20, 40, 1000]).range([0, 3, 11]);
    return Math.ceil(yTickScale(length));
  }
  getLength(el) {
    if (this.axisConfig.isHorizontal()) {
      return (0, _jquery.default)(el).width();
    } else {
      return (0, _jquery.default)(el).height();
    }
  }
  adjustSize() {
    const config = this.axisConfig;
    const style = config.get('style');
    const chartEl = this.visConfig.get('el');
    const position = config.get('position');
    const axisPadding = 5;
    return function (selection) {
      const text = selection.selectAll('.tick text');
      const lengths = [];
      text.each(function textWidths() {
        lengths.push((() => {
          if (config.isHorizontal()) {
            return _d.default.select(this.parentNode).node().getBBox().height;
          } else {
            return _d.default.select(this.parentNode).node().getBBox().width;
          }
        })());
      });
      let length = lengths.length > 0 ? _lodash.default.max(lengths) : 0;
      length += axisPadding;
      if (config.isHorizontal()) {
        selection.attr('height', Math.ceil(length));
        if (position === 'top') {
          selection.select('g').attr('transform', `translate(0, ${length - parseInt(style.lineWidth)})`);
          selection.select('path').attr('transform', 'translate(1,0)');
        }
        if (config.get('type') === 'value') {
          const spacerNodes = (0, _jquery.default)(chartEl).find(`.visAxis__spacer--y-${position}`);
          const elHeight = (0, _jquery.default)(chartEl).find(`.visAxis__column--${position}`).height();
          spacerNodes.height(elHeight);
        }
      } else {
        const axisWidth = Math.ceil(length);
        selection.attr('width', axisWidth);
        if (position === 'left') {
          selection.select('g').attr('transform', `translate(${axisWidth},0)`);
        }
      }
    };
  }
  validate() {
    if (this.axisConfig.isLogScale() && this.axisConfig.isPercentage()) {
      throw new _errors.VislibError(`Can't mix percentage mode with log scale.`);
    }
  }
  draw() {
    const svgs = [];
    const self = this;
    const config = this.axisConfig;
    const style = config.get('style');
    return function (selection) {
      const n = selection[0].length;
      if (config.get('show') && self.axisTitle && ['left', 'top'].includes(config.get('position'))) {
        self.axisTitle.render(selection);
      }
      selection.each(function () {
        const el = this;
        const div = _d.default.select(el);
        const width = (0, _jquery.default)(el).width();
        const height = (0, _jquery.default)(el).height();
        const length = self.getLength(el, n);
        self.validate();
        const axis = self.getAxis(length);
        if (config.get('show')) {
          const svg = div.append('svg').attr('focusable', 'false').attr('width', width).attr('height', height);
          svgs.push(svg);
          const axisClass = self.axisConfig.isHorizontal() ? 'x' : 'y';
          svg.append('g').attr('class', `${axisClass} axis ${config.get('id')}`).call(axis);
          const container = svg.select('g.axis').node();
          if (container) {
            svg.select('path').style('stroke', style.color).style('stroke-width', style.lineWidth).style('stroke-opacity', style.opacity);
            svg.selectAll('line').style('stroke', style.tickColor).style('stroke-width', style.tickWidth).style('stroke-opacity', style.opacity);
          }
          if (self.axisLabels) self.axisLabels.render(svg);
        }
      });
      if (self.axisTitle && ['right', 'bottom'].includes(config.get('position'))) {
        self.axisTitle.render(selection);
      }
      svgs.forEach(svg => svg.call(self.adjustSize()));
    };
  }
}
exports.Axis = Axis;