"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getEventVariableList = exports.getEventScopeValues = void 0;
var _i18n = require("@kbn/i18n");
var _monaco = require("@kbn/monaco");
var _public = require("@kbn/embeddable-plugin/public");
var _public2 = require("@kbn/ui-actions-plugin/public");
var _util = require("./util");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getEventScopeFromRangeSelectTriggerContext = eventScopeInput => {
  const {
    table,
    column: columnIndex,
    range
  } = eventScopeInput.data;
  const column = table.columns[columnIndex];
  return (0, _util.deleteUndefinedKeys)({
    key: (0, _util.toPrimitiveOrUndefined)(column === null || column === void 0 ? void 0 : column.meta.field),
    from: (0, _util.toPrimitiveOrUndefined)(range[0]),
    to: (0, _util.toPrimitiveOrUndefined)(range[range.length - 1])
  });
};
const getEventScopeFromValueClickTriggerContext = eventScopeInput => {
  var _eventScopeInput$data, _points$, _points$2;
  const negate = (_eventScopeInput$data = eventScopeInput.data.negate) !== null && _eventScopeInput$data !== void 0 ? _eventScopeInput$data : false;
  const points = eventScopeInput.data.data.map(({
    table,
    value,
    column: columnIndex
  }) => {
    var _column$meta;
    const column = table.columns[columnIndex];
    return {
      value: (0, _util.toPrimitiveOrUndefined)(value),
      key: column === null || column === void 0 ? void 0 : (_column$meta = column.meta) === null || _column$meta === void 0 ? void 0 : _column$meta.field
    };
  });
  return (0, _util.deleteUndefinedKeys)({
    key: (_points$ = points[0]) === null || _points$ === void 0 ? void 0 : _points$.key,
    value: (_points$2 = points[0]) === null || _points$2 === void 0 ? void 0 : _points$2.value,
    negate,
    points
  });
};
const getEventScopeFromRowClickTriggerContext = ctx => {
  const {
    data
  } = ctx;
  const embeddable = ctx.embeddable;
  const {
    rowIndex
  } = data;
  const columns = data.columns || data.table.columns.map(({
    id
  }) => id);
  const values = [];
  const keys = [];
  const columnNames = [];
  const row = data.table.rows[rowIndex];
  for (const columnId of columns) {
    const column = data.table.columns.find(({
      id
    }) => id === columnId);
    if (!column) {
      // This should never happe, but in case it does we log data necessary for debugging.
      // eslint-disable-next-line no-console
      console.error(data, embeddable ? `Embeddable [${embeddable.getTitle()}]` : null);
      throw new Error('Could not find a datatable column.');
    }
    values.push(row[columnId]);
    keys.push(column.meta.field || '');
    columnNames.push(column.name || column.meta.field || '');
  }
  const scope = {
    rowIndex,
    values,
    keys,
    columnNames
  };
  return scope;
};
const getEventScopeValues = eventScopeInput => {
  if ((0, _public.isRangeSelectTriggerContext)(eventScopeInput)) {
    return getEventScopeFromRangeSelectTriggerContext(eventScopeInput);
  } else if ((0, _public.isValueClickTriggerContext)(eventScopeInput)) {
    return getEventScopeFromValueClickTriggerContext(eventScopeInput);
  } else if ((0, _public.isRowClickTriggerContext)(eventScopeInput)) {
    return getEventScopeFromRowClickTriggerContext(eventScopeInput);
  } else if ((0, _public.isContextMenuTriggerContext)(eventScopeInput)) {
    return {};
  } else {
    throw new Error("UrlDrilldown [getEventScope] can't build scope from not supported trigger");
  }
};
exports.getEventScopeValues = getEventScopeValues;
const kind = _monaco.monaco.languages.CompletionItemKind.Event;
const sortPrefix = '1.';
const valueClickVariables = [{
  label: 'event.value',
  sortText: sortPrefix + 'event.value',
  title: _i18n.i18n.translate('xpack.urlDrilldown.click.event.value.title', {
    defaultMessage: 'Click value.'
  }),
  documentation: _i18n.i18n.translate('xpack.urlDrilldown.click.event.value.documentation', {
    defaultMessage: 'Value behind clicked data point.'
  }),
  kind
}, {
  label: 'event.key',
  sortText: sortPrefix + 'event.key',
  title: _i18n.i18n.translate('xpack.urlDrilldown.click.event.key.title', {
    defaultMessage: 'Name of clicked field.'
  }),
  documentation: _i18n.i18n.translate('xpack.urlDrilldown.click.event.key.documentation', {
    defaultMessage: 'Field name behind clicked data point.'
  }),
  kind
}, {
  label: 'event.negate',
  sortText: sortPrefix + 'event.negate',
  title: _i18n.i18n.translate('xpack.urlDrilldown.click.event.negate.title', {
    defaultMessage: 'Whether the filter is negated.'
  }),
  documentation: _i18n.i18n.translate('xpack.urlDrilldown.click.event.negate.documentation', {
    defaultMessage: 'Boolean, indicating whether clicked data point resulted in negative filter.'
  }),
  kind
}, {
  label: 'event.points',
  sortText: sortPrefix + 'event.points',
  title: _i18n.i18n.translate('xpack.urlDrilldown.click.event.points.title', {
    defaultMessage: 'List of all clicked points.'
  }),
  documentation: _i18n.i18n.translate('xpack.urlDrilldown.click.event.points.documentation', {
    defaultMessage: 'Some visualizations have clickable points that emit more than one data point. Use list of data points in case a single value is insufficient.'
  }),
  kind
}];
const rowClickVariables = [{
  label: 'event.values',
  sortText: sortPrefix + 'event.values',
  title: _i18n.i18n.translate('xpack.urlDrilldown.row.event.values.title', {
    defaultMessage: 'List of row cell values.'
  }),
  documentation: _i18n.i18n.translate('xpack.urlDrilldown.row.event.values.documentation', {
    defaultMessage: 'An array of all cell values for the raw on which the action will execute.'
  }),
  kind
}, {
  label: 'event.keys',
  sortText: sortPrefix + 'event.keys',
  title: _i18n.i18n.translate('xpack.urlDrilldown.row.event.keys.title', {
    defaultMessage: 'List of row cell fields.'
  }),
  documentation: _i18n.i18n.translate('xpack.urlDrilldown.row.event.keys.documentation', {
    defaultMessage: 'An array of field names for each column.'
  }),
  kind
}, {
  label: 'event.columnNames',
  sortText: sortPrefix + 'event.columnNames',
  title: _i18n.i18n.translate('xpack.urlDrilldown.row.event.columnNames.title', {
    defaultMessage: 'List of table column names.'
  }),
  documentation: _i18n.i18n.translate('xpack.urlDrilldown.row.event.columnNames.documentation', {
    defaultMessage: 'An array of column names.'
  }),
  kind
}, {
  label: 'event.rowIndex',
  sortText: sortPrefix + 'event.rowIndex',
  title: _i18n.i18n.translate('xpack.urlDrilldown.row.event.rowIndex.title', {
    defaultMessage: 'Clicked row index.'
  }),
  documentation: _i18n.i18n.translate('xpack.urlDrilldown.row.event.rowIndex.documentation', {
    defaultMessage: 'Number, representing the row that was clicked, starting from 0.'
  }),
  kind
}];
const selectRangeVariables = [{
  label: 'event.key',
  sortText: sortPrefix + 'event.key',
  title: _i18n.i18n.translate('xpack.urlDrilldown.range.event.key.title', {
    defaultMessage: 'Name of aggregation field.'
  }),
  documentation: _i18n.i18n.translate('xpack.urlDrilldown.range.event.key.documentation', {
    defaultMessage: 'Aggregation field behind the selected range, if available.'
  }),
  kind
}, {
  label: 'event.from',
  sortText: sortPrefix + 'event.from',
  title: _i18n.i18n.translate('xpack.urlDrilldown.range.event.from.title', {
    defaultMessage: 'Range start value.'
  }),
  documentation: _i18n.i18n.translate('xpack.urlDrilldown.range.event.from.documentation', {
    defaultMessage: '`from` value of the selected range. Depending on your data, could be either a date or number.'
  }),
  kind
}, {
  label: 'event.to',
  sortText: sortPrefix + 'event.to',
  title: _i18n.i18n.translate('xpack.urlDrilldown.range.event.to.title', {
    defaultMessage: 'Range end value.'
  }),
  documentation: _i18n.i18n.translate('xpack.urlDrilldown.range.event.to.documentation', {
    defaultMessage: '`to` value of the selected range. Depending on your data, could be either a date or number.'
  }),
  kind
}];
const getEventVariableList = context => {
  const [trigger] = context.triggers;
  switch (trigger) {
    case _public.VALUE_CLICK_TRIGGER:
      return [...valueClickVariables];
    case _public2.ROW_CLICK_TRIGGER:
      return [...rowClickVariables];
    case _public.SELECT_RANGE_TRIGGER:
      return [...selectRangeVariables];
  }
  return [];
};
exports.getEventVariableList = getEventVariableList;