"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = exports.RuleEdit = void 0;
var _react = _interopRequireWildcard(require("react"));
var _i18nReact = require("@kbn/i18n-react");
var _common = require("@kbn/alerting-plugin/common");
var _eui = require("@elastic/eui");
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
var _types = require("../../../types");
var _rule_form = require("./rule_form");
var _rule_errors = require("./rule_errors");
var _rule_reducer = require("./rule_reducer");
var _update = require("../../lib/rule_api/update");
var _rule_types = require("../../lib/rule_api/rule_types");
var _health_check = require("../../components/health_check");
var _health_context = require("../../context/health_context");
var _kibana = require("../../../common/lib/kibana");
var _confirm_rule_close = require("./confirm_rule_close");
var _has_rule_changed = require("./has_rule_changed");
var _value_validators = require("../../lib/value_validators");
var _config_api = require("../../../common/lib/config_api");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const cloneAndMigrateRule = initialRule => {
  const clonedRule = (0, _lodash.cloneDeep)((0, _lodash.omit)(initialRule, 'notifyWhen', 'throttle'));
  const hasRuleLevelNotifyWhen = Boolean(initialRule.notifyWhen);
  const hasRuleLevelThrottle = Boolean(initialRule.throttle);
  if (hasRuleLevelNotifyWhen || hasRuleLevelThrottle) {
    const frequency = hasRuleLevelNotifyWhen ? {
      summary: false,
      notifyWhen: initialRule.notifyWhen,
      throttle: initialRule.notifyWhen === _common.RuleNotifyWhen.THROTTLE ? initialRule.throttle : null
    } : {
      summary: false,
      notifyWhen: _common.RuleNotifyWhen.THROTTLE,
      throttle: initialRule.throttle
    };
    clonedRule.actions = clonedRule.actions.map(action => ({
      ...action,
      frequency
    }));
  }
  return clonedRule;
};
const RuleEdit = ({
  initialRule,
  onClose,
  reloadRules,
  onSave,
  hideInterval,
  ruleTypeRegistry,
  actionTypeRegistry,
  metadata: initialMetadata,
  ...props
}) => {
  const onSaveHandler = onSave !== null && onSave !== void 0 ? onSave : reloadRules;
  const [{
    rule
  }, dispatch] = (0, _react.useReducer)(_rule_reducer.ruleReducer, {
    rule: cloneAndMigrateRule(initialRule)
  });
  const [isSaving, setIsSaving] = (0, _react.useState)(false);
  const [hasActionsDisabled, setHasActionsDisabled] = (0, _react.useState)(false);
  const [hasActionsWithBrokenConnector, setHasActionsWithBrokenConnector] = (0, _react.useState)(false);
  const [isConfirmRuleCloseModalOpen, setIsConfirmRuleCloseModalOpen] = (0, _react.useState)(false);
  const [ruleActionsErrors, setRuleActionsErrors] = (0, _react.useState)([]);
  const [isLoading, setIsLoading] = (0, _react.useState)(false);
  const [serverRuleType, setServerRuleType] = (0, _react.useState)(props.ruleType);
  const [config, setConfig] = (0, _react.useState)({
    isUsingSecurity: false
  });
  const [metadata, setMetadata] = (0, _react.useState)(initialMetadata);
  const onChangeMetaData = (0, _react.useCallback)(newMetadata => setMetadata(newMetadata), []);
  const {
    http,
    notifications: {
      toasts
    }
  } = (0, _kibana.useKibana)().services;
  const setRule = value => {
    dispatch({
      command: {
        type: 'setRule'
      },
      payload: {
        key: 'rule',
        value
      }
    });
  };
  const ruleType = ruleTypeRegistry.get(rule.ruleTypeId);
  (0, _react.useEffect)(() => {
    (async () => {
      setConfig(await (0, _config_api.triggersActionsUiConfig)({
        http
      }));
    })();
  }, [http]);
  (0, _react.useEffect)(() => {
    (async () => {
      setIsLoading(true);
      const res = await (0, _rule_errors.getRuleActionErrors)(rule.actions, actionTypeRegistry);
      setRuleActionsErrors([...res]);
      setIsLoading(false);
    })();
  }, [rule.actions, actionTypeRegistry]);
  (0, _react.useEffect)(() => {
    if (!props.ruleType && !serverRuleType) {
      (async () => {
        const serverRuleTypes = await (0, _rule_types.loadRuleTypes)({
          http
        });
        for (const _serverRuleType of serverRuleTypes) {
          if (ruleType.id === _serverRuleType.id) {
            setServerRuleType(_serverRuleType);
          }
        }
      })();
    }
  }, [props.ruleType, ruleType.id, serverRuleType, http]);
  const {
    ruleBaseErrors,
    ruleErrors,
    ruleParamsErrors
  } = (0, _rule_errors.getRuleErrors)(rule, ruleType, config);
  const checkForChangesAndCloseFlyout = () => {
    if ((0, _has_rule_changed.hasRuleChanged)(rule, initialRule, true)) {
      setIsConfirmRuleCloseModalOpen(true);
    } else {
      onClose(_types.RuleFlyoutCloseReason.CANCELED, metadata);
    }
  };
  async function onSaveRule() {
    setIsSaving(true);
    try {
      if (!isLoading && (0, _rule_errors.isValidRule)(rule, ruleErrors, ruleActionsErrors) && !hasActionsWithBrokenConnector) {
        const newRule = await (0, _update.updateRule)({
          http,
          rule,
          id: rule.id
        });
        toasts.addSuccess(_i18n.i18n.translate('xpack.triggersActionsUI.sections.ruleEdit.saveSuccessNotificationText', {
          defaultMessage: "Updated '{ruleName}'",
          values: {
            ruleName: newRule.name
          }
        }));
        onClose(_types.RuleFlyoutCloseReason.SAVED, metadata);
        if (onSaveHandler) {
          onSaveHandler(metadata);
        }
      } else {
        setRule((0, _value_validators.getRuleWithInvalidatedFields)(rule, ruleParamsErrors, ruleBaseErrors, ruleActionsErrors));
      }
    } catch (errorRes) {
      var _errorRes$body$messag, _errorRes$body;
      toasts.addDanger((_errorRes$body$messag = (_errorRes$body = errorRes.body) === null || _errorRes$body === void 0 ? void 0 : _errorRes$body.message) !== null && _errorRes$body$messag !== void 0 ? _errorRes$body$messag : _i18n.i18n.translate('xpack.triggersActionsUI.sections.ruleEdit.saveErrorNotificationText', {
        defaultMessage: 'Cannot update rule.'
      }));
    }
    setIsSaving(false);
  }
  return /*#__PURE__*/_react.default.createElement(_eui.EuiPortal, null, /*#__PURE__*/_react.default.createElement(_eui.EuiFlyout, {
    onClose: checkForChangesAndCloseFlyout,
    "aria-labelledby": "flyoutRuleEditTitle",
    size: "m",
    maxWidth: 620,
    ownFocus: false
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlyoutHeader, {
    hasBorder: true
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiTitle, {
    size: "s",
    "data-test-subj": "editRuleFlyoutTitle"
  }, /*#__PURE__*/_react.default.createElement("h3", {
    id: "flyoutTitle"
  }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
    defaultMessage: "Edit rule",
    id: "xpack.triggersActionsUI.sections.ruleEdit.flyoutTitle"
  })))), /*#__PURE__*/_react.default.createElement(_health_context.HealthContextProvider, null, /*#__PURE__*/_react.default.createElement(_health_check.HealthCheck, {
    inFlyout: true,
    waitForCheck: true
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlyoutBody, null, hasActionsDisabled && /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement(_eui.EuiCallOut, {
    size: "s",
    color: "danger",
    iconType: "rule",
    "data-test-subj": "hasActionsDisabled",
    title: _i18n.i18n.translate('xpack.triggersActionsUI.sections.ruleEdit.disabledActionsWarningTitle', {
      defaultMessage: 'This rule has actions that are disabled'
    })
  }), /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, null)), /*#__PURE__*/_react.default.createElement(_rule_form.RuleForm, {
    rule: rule,
    config: config,
    dispatch: dispatch,
    errors: ruleErrors,
    actionTypeRegistry: actionTypeRegistry,
    hideInterval: hideInterval,
    ruleTypeRegistry: ruleTypeRegistry,
    canChangeTrigger: false,
    setHasActionsDisabled: setHasActionsDisabled,
    setHasActionsWithBrokenConnector: setHasActionsWithBrokenConnector,
    operation: _i18n.i18n.translate('xpack.triggersActionsUI.sections.ruleEdit.operationName', {
      defaultMessage: 'edit'
    }),
    metadata: metadata,
    onChangeMetaData: onChangeMetaData
  })), /*#__PURE__*/_react.default.createElement(_eui.EuiFlyoutFooter, null, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexGroup, {
    justifyContent: "spaceBetween"
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
    grow: false
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiButtonEmpty, {
    "data-test-subj": "cancelSaveEditedRuleButton",
    onClick: () => checkForChangesAndCloseFlyout()
  }, _i18n.i18n.translate('xpack.triggersActionsUI.sections.ruleEdit.cancelButtonLabel', {
    defaultMessage: 'Cancel'
  }))), isLoading ? /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
    grow: false
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
    size: "s"
  }), /*#__PURE__*/_react.default.createElement(_eui.EuiLoadingSpinner, {
    size: "l"
  })) : /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null), /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
    grow: false
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexGroup, {
    alignItems: "center"
  }, config.isUsingSecurity && /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
    grow: false
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiIconTip, {
    type: "warning",
    position: "top",
    "data-test-subj": "changeInPrivilegesTip",
    content: _i18n.i18n.translate('xpack.triggersActionsUI.sections.ruleEdit.changeInPrivilegesLabel', {
      defaultMessage: 'Saving this rule will change its privileges and might change its behavior.'
    })
  })), /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
    grow: false
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiButton, {
    fill: true,
    color: "success",
    "data-test-subj": "saveEditedRuleButton",
    type: "submit",
    iconType: "check",
    isLoading: isSaving,
    onClick: async () => await onSaveRule()
  }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
    id: "xpack.triggersActionsUI.sections.ruleEdit.saveButtonLabel",
    defaultMessage: "Save"
  }))))))))), isConfirmRuleCloseModalOpen && /*#__PURE__*/_react.default.createElement(_confirm_rule_close.ConfirmRuleClose, {
    onConfirm: () => {
      setIsConfirmRuleCloseModalOpen(false);
      onClose(_types.RuleFlyoutCloseReason.CANCELED, metadata);
    },
    onCancel: () => {
      setIsConfirmRuleCloseModalOpen(false);
    }
  })));
};

// eslint-disable-next-line import/no-default-export
exports.default = exports.RuleEdit = RuleEdit;