"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useGetTransforms = void 0;
var _mlIsDefined = require("@kbn/ml-is-defined");
var _type_guards = require("../../../common/api_schemas/type_guards");
var _constants = require("../../../common/constants");
var _transform_stats = require("../../../common/types/transform_stats");
var _common = require("../common");
var _use_api = require("./use_api");
var _transform = require("../common/transform");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useGetTransforms = (setTransforms, setTransformNodes, setErrorMessage, setTransformIdsWithoutConfig, setIsInitialized, blockRefresh) => {
  const api = (0, _use_api.useApi)();
  let concurrentLoads = 0;
  const getTransforms = async (forceRefresh = false) => {
    if (forceRefresh === true || blockRefresh === false) {
      _common.refreshTransformList$.next(_common.REFRESH_TRANSFORM_LIST_STATE.LOADING);
      concurrentLoads++;
      if (concurrentLoads > 1) {
        return;
      }
      const fetchOptions = {
        asSystemRequest: true
      };
      const transformNodes = await api.getTransformNodes();
      const transformConfigs = await api.getTransforms(fetchOptions);
      const transformStats = await api.getTransformsStats(fetchOptions);
      if (!(0, _type_guards.isGetTransformsResponseSchema)(transformConfigs) || !(0, _type_guards.isGetTransformsStatsResponseSchema)(transformStats) || !(0, _type_guards.isGetTransformNodesResponseSchema)(transformNodes)) {
        // An error is followed immediately by setting the state to idle.
        // This way we're able to treat ERROR as a one-time-event like REFRESH.
        _common.refreshTransformList$.next(_common.REFRESH_TRANSFORM_LIST_STATE.ERROR);
        _common.refreshTransformList$.next(_common.REFRESH_TRANSFORM_LIST_STATE.IDLE);
        setTransformNodes(0);
        setTransforms([]);
        setIsInitialized(true);
        if (!(0, _type_guards.isGetTransformsResponseSchema)(transformConfigs)) {
          setErrorMessage(transformConfigs);
        } else if (!(0, _type_guards.isGetTransformsStatsResponseSchema)(transformStats)) {
          setErrorMessage(transformStats);
        }
        return;
      }

      // There might be some errors with fetching certain transforms
      // For example, when task exists and is running but the config is deleted
      if (Array.isArray(transformConfigs.errors) && transformConfigs.errors.length > 0) {
        const danglingTaskIdMatches = transformConfigs.errors.filter(e => e.type === _transform.TRANSFORM_ERROR_TYPE.DANGLING_TASK).map(e => {
          // Getting the transform id from the ES error message
          const matches = /\[([^)]+)\]/.exec(e.reason);
          return Array.isArray(matches) && matches.length >= 1 ? matches[1] : undefined;
        }).filter(_mlIsDefined.isDefined);
        setTransformIdsWithoutConfig(danglingTaskIdMatches.length > 0 ? danglingTaskIdMatches : undefined);
      } else {
        setTransformIdsWithoutConfig(undefined);
      }
      const tableRows = transformConfigs.transforms.reduce((reducedtableRows, config) => {
        const stats = (0, _type_guards.isGetTransformsStatsResponseSchema)(transformStats) ? transformStats.transforms.find(d => config.id === d.id) : undefined;

        // A newly created transform might not have corresponding stats yet.
        // If that's the case we just skip the transform and don't add it to the transform list yet.
        if (!(0, _transform_stats.isTransformStats)(stats)) {
          return reducedtableRows;
        }

        // Table with expandable rows requires `id` on the outer most level
        reducedtableRows.push({
          id: config.id,
          config,
          mode: typeof config.sync !== 'undefined' ? _constants.TRANSFORM_MODE.CONTINUOUS : _constants.TRANSFORM_MODE.BATCH,
          stats,
          alerting_rules: config.alerting_rules
        });
        return reducedtableRows;
      }, []);
      setTransformNodes(transformNodes.count);
      setTransforms(tableRows);
      setErrorMessage(undefined);
      setIsInitialized(true);
      _common.refreshTransformList$.next(_common.REFRESH_TRANSFORM_LIST_STATE.IDLE);
      concurrentLoads--;
      if (concurrentLoads > 0) {
        concurrentLoads = 0;
        getTransforms(true);
        return;
      }
    }
  };
  return getTransforms;
};
exports.useGetTransforms = useGetTransforms;