"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SpacesManager = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _operators = require("rxjs/operators");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TAG_TYPE = 'tag';
class SpacesManager {
  constructor(http) {
    (0, _defineProperty2.default)(this, "activeSpace$", new _rxjs.BehaviorSubject(null));
    (0, _defineProperty2.default)(this, "serverBasePath", void 0);
    (0, _defineProperty2.default)(this, "_onActiveSpaceChange$", void 0);
    this.http = http;
    this.serverBasePath = http.basePath.serverBasePath;
    this._onActiveSpaceChange$ = this.activeSpace$.asObservable().pipe((0, _operators.skipWhile)(v => v == null));
  }
  get onActiveSpaceChange$() {
    if (!this.activeSpace$.value) {
      this.refreshActiveSpace();
    }
    return this._onActiveSpaceChange$;
  }
  async getSpaces(options = {}) {
    const {
      purpose,
      includeAuthorizedPurposes
    } = options;
    const query = {
      purpose,
      include_authorized_purposes: includeAuthorizedPurposes
    };
    return await this.http.get('/api/spaces/space', {
      query
    });
  }
  async getSpace(id) {
    return await this.http.get(`/api/spaces/space/${encodeURIComponent(id)}`);
  }
  async getActiveSpace({
    forceRefresh = false
  } = {}) {
    if (this.isAnonymousPath()) {
      throw new Error(`Cannot retrieve the active space for anonymous paths`);
    }
    if (forceRefresh || !this.activeSpace$.value) {
      await this.refreshActiveSpace();
    }
    return this.activeSpace$.value;
  }
  async createSpace(space) {
    await this.http.post(`/api/spaces/space`, {
      body: JSON.stringify(space)
    });
  }
  async updateSpace(space) {
    await this.http.put(`/api/spaces/space/${encodeURIComponent(space.id)}`, {
      query: {
        overwrite: true
      },
      body: JSON.stringify(space)
    });
    const activeSpaceId = (await this.getActiveSpace()).id;
    if (space.id === activeSpaceId) {
      this.refreshActiveSpace();
    }
  }
  async deleteSpace(space) {
    await this.http.delete(`/api/spaces/space/${encodeURIComponent(space.id)}`);
  }
  async disableLegacyUrlAliases(aliases) {
    await this.http.post('/api/spaces/_disable_legacy_url_aliases', {
      body: JSON.stringify({
        aliases
      })
    });
  }
  async copySavedObjects(objects, spaces, includeReferences, createNewCopies, overwrite) {
    return this.http.post('/api/spaces/_copy_saved_objects', {
      body: JSON.stringify({
        objects,
        spaces,
        includeReferences,
        createNewCopies,
        ...(createNewCopies ? {
          overwrite: false
        } : {
          overwrite
        }) // ignore the overwrite option if createNewCopies is enabled
      })
    });
  }

  async resolveCopySavedObjectsErrors(objects, retries, includeReferences, createNewCopies) {
    return this.http.post(`/api/spaces/_resolve_copy_saved_objects_errors`, {
      body: JSON.stringify({
        objects,
        includeReferences,
        createNewCopies,
        retries
      })
    });
  }
  async getShareSavedObjectPermissions(type) {
    return this.http.get('/internal/security/_share_saved_object_permissions', {
      query: {
        type
      }
    }).catch(err => {
      var _err$body;
      const isNotFound = (err === null || err === void 0 ? void 0 : (_err$body = err.body) === null || _err$body === void 0 ? void 0 : _err$body.statusCode) === 404;
      if (isNotFound) {
        // security is not enabled
        return {
          shareToAllSpaces: true
        };
      }
      throw err;
    });
  }
  async getShareableReferences(objects) {
    const response = await this.http.post(`/api/spaces/_get_shareable_references`, {
      body: JSON.stringify({
        objects
      })
    });

    // We should exclude any child-reference tags because we don't yet support reconciling/merging duplicate tags. In other words: tags can
    // be shared directly, but if a tag is only included as a reference of a requested object, it should not be shared.
    const requestedObjectsSet = objects.reduce((acc, {
      type,
      id
    }) => acc.add(`${type}:${id}`), new Set());
    const filteredObjects = response.objects.filter(({
      type,
      id
    }) => type !== TAG_TYPE || requestedObjectsSet.has(`${type}:${id}`));
    return {
      objects: filteredObjects
    };
  }
  async updateSavedObjectsSpaces(objects, spacesToAdd, spacesToRemove) {
    return this.http.post(`/api/spaces/_update_objects_spaces`, {
      body: JSON.stringify({
        objects,
        spacesToAdd,
        spacesToRemove
      })
    });
  }
  redirectToSpaceSelector() {
    window.location.href = `${this.serverBasePath}/spaces/space_selector`;
  }
  async refreshActiveSpace() {
    // Anonymous paths (such as login/logout) should not request the active space under any circumstances.
    if (this.isAnonymousPath()) {
      return;
    }
    const activeSpace = await this.http.get('/internal/spaces/_active_space');
    this.activeSpace$.next(activeSpace);
  }
  isAnonymousPath() {
    return this.http.anonymousPaths.isAnonymous(window.location.pathname);
  }
}
exports.SpacesManager = SpacesManager;