"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useFieldBrowserOptions = void 0;
var _react = require("react");
var _use_data_view = require("../../../common/containers/source/use_data_view");
var _use_selector = require("../../../common/hooks/use_selector");
var _kibana = require("../../../common/lib/kibana");
var _store = require("../../../common/store");
var _default_headers = require("../timeline/body/column_headers/default_headers");
var _constants = require("../timeline/body/constants");
var _create_field_button = require("./create_field_button");
var _field_table_columns = require("./field_table_columns");
var _use_start_transaction = require("../../../common/lib/apm/use_start_transaction");
var _user_actions = require("../../../common/lib/apm/user_actions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useFieldBrowserOptions = ({
  sourcererScope,
  editorActionsRef,
  removeColumn,
  upsertColumn
}) => {
  const [dataView, setDataView] = (0, _react.useState)(null);
  const {
    startTransaction
  } = (0, _use_start_transaction.useStartTransaction)();
  const {
    indexFieldsSearch
  } = (0, _use_data_view.useDataView)();
  const {
    dataViewFieldEditor,
    data: {
      dataViews
    }
  } = (0, _kibana.useKibana)().services;
  const scopeIdSelector = (0, _react.useMemo)(() => _store.sourcererSelectors.scopeIdSelector(), []);
  const {
    missingPatterns,
    selectedDataViewId
  } = (0, _use_selector.useDeepEqualSelector)(state => scopeIdSelector(state, sourcererScope));
  (0, _react.useEffect)(() => {
    const fetchAndSetDataView = async dataViewId => {
      const aDatView = await dataViews.get(dataViewId);
      setDataView(aDatView);
    };
    if (selectedDataViewId != null && !missingPatterns.length) {
      fetchAndSetDataView(selectedDataViewId);
    }
  }, [selectedDataViewId, missingPatterns, dataViews]);
  const openFieldEditor = (0, _react.useCallback)(fieldName => {
    if (dataView && selectedDataViewId) {
      const closeFieldEditor = dataViewFieldEditor.openEditor({
        ctx: {
          dataView
        },
        fieldName,
        onSave: async savedFields => {
          startTransaction({
            name: _user_actions.FIELD_BROWSER_ACTIONS.FIELD_SAVED
          });
          // Fetch the updated list of fields
          // Using cleanCache since the number of fields might have not changed, but we need to update the state anyway
          await indexFieldsSearch({
            dataViewId: selectedDataViewId,
            cleanCache: true
          });
          for (const savedField of savedFields) {
            if (fieldName && fieldName !== savedField.name) {
              // Remove old field from event table when renaming a field
              removeColumn(fieldName);
            }

            // Add the saved column field to the table in any case

            upsertColumn({
              columnHeaderType: _default_headers.defaultColumnHeaderType,
              id: savedField.name,
              initialWidth: _constants.DEFAULT_COLUMN_MIN_WIDTH
            }, 0);
          }
          if (editorActionsRef) {
            editorActionsRef.current = null;
          }
        }
      });
      if (editorActionsRef) {
        editorActionsRef.current = {
          closeEditor: () => {
            editorActionsRef.current = null;
            closeFieldEditor();
          }
        };
      }
    }
  }, [dataView, selectedDataViewId, dataViewFieldEditor, editorActionsRef, indexFieldsSearch, removeColumn, upsertColumn, startTransaction]);
  const openDeleteFieldModal = (0, _react.useCallback)(fieldName => {
    if (dataView && selectedDataViewId) {
      dataViewFieldEditor.openDeleteModal({
        ctx: {
          dataView
        },
        fieldName,
        onDelete: async () => {
          startTransaction({
            name: _user_actions.FIELD_BROWSER_ACTIONS.FIELD_DELETED
          });

          // Fetch the updated list of fields
          await indexFieldsSearch({
            dataViewId: selectedDataViewId
          });
          removeColumn(fieldName);
        }
      });
    }
  }, [dataView, selectedDataViewId, dataViewFieldEditor, indexFieldsSearch, removeColumn, startTransaction]);
  const hasFieldEditPermission = (0, _react.useMemo)(() => dataViewFieldEditor === null || dataViewFieldEditor === void 0 ? void 0 : dataViewFieldEditor.userPermissions.editIndexPattern(), [dataViewFieldEditor === null || dataViewFieldEditor === void 0 ? void 0 : dataViewFieldEditor.userPermissions]);
  const createFieldButton = (0, _create_field_button.useCreateFieldButton)({
    isAllowed: hasFieldEditPermission && !!selectedDataViewId,
    loading: !dataView,
    openFieldEditor
  });
  const getFieldTableColumns = (0, _field_table_columns.useFieldTableColumns)({
    hasFieldEditPermission,
    openFieldEditor,
    openDeleteFieldModal
  });
  const memoized = (0, _react.useMemo)(() => ({
    createFieldButton,
    getFieldTableColumns
  }), [createFieldButton, getFieldTableColumns]);
  return memoized;
};
exports.useFieldBrowserOptions = useFieldBrowserOptions;