"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.array = array;
exports.literal = literal;
exports.number = anyNumber;
exports.object = object;
exports.oneOf = oneOf;
exports.string = anyString;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Validate that `value` matches at least one of `validators`.
 * Use this to create a predicate for a union type.
 * e.g.
 * ```
 * import * as schema from './schema';
 * const isAscOrDesc: (value: unknown) => value is 'asc' | 'desc' = schema.oneOf([
 *   schema.literal('asc' as const),
 *   schema.literal('desc' as const),
 * ]);
 * ```
 */
function oneOf(validators) {
  return function (value) {
    for (const validator of validators) {
      if (validator(value)) {
        return true;
      }
    }
    return false;
  };
}

/**
 * Validate that `value` is an array and that each of its elements matches `elementValidator`.
 * Use this to create a predicate for an array type.
 * ```
 * import * as schema from './schema';
 * const isAscOrDesc: (value: unknown) => value is 'asc' | 'desc' = schema.oneOf([
 *   schema.literal('asc' as const),
 *   schema.literal('desc' as const),
 * ]);
 * ```
 */
function array(elementValidator) {
  return function (value) {
    if (Array.isArray(value)) {
      for (const element of value) {
        const result = elementValidator(element);
        if (!result) {
          return false;
        }
      }
      return true;
    }
    return false;
  };
}

/**
 * The keys of `T` where `undefined` is assignable to the corresponding value.
 * Used to figure out which keys could be made optional.
 */

/**
 * Validate that `value` is an object with string keys. The value at each key is tested against its own validator.
 *
 * Use this to create a predicate for a type like `{ a: string[] }`. For example:
 * ```ts
 * import * as schema from './schema';
 * const myValidator: (value: unknown) => value is { a: string[] } = schema.object({
 *   a: schema.array(schema.string()),
 * });
 * ```
 */
function object(validatorDictionary) {
  return function (value) {
    // This only validates non-null objects
    if (typeof value !== 'object' || value === null) {
      return false;
    }

    // Rebind value as the result type so that we can interrogate it
    const trusted = value;

    // Get each validator in the validator dictionary and use it to validate the corresponding value
    for (const key of Object.keys(validatorDictionary)) {
      const validator = validatorDictionary[key];
      if (!validator(trusted[key])) {
        return false;
      }
    }
    return true;
  };
}

/**
 * Validate that `value` is strictly equal to `acceptedValue`.
 * Use this for a literal type, for example:
 * ```
 * import * as schema from './schema';
 * const isAscOrDesc: (value: unknown) => value is 'asc' | 'desc' = schema.oneOf([
 *   schema.literal('asc' as const),
 *   schema.literal('desc' as const),
 * ]);
 * ```
 */
function literal(acceptedValue) {
  return function (value) {
    return acceptedValue === value;
  };
}

/**
 * Validate that `value` is a string.
 * NB: this is used as `string` externally via named export.
 * Usage:
 * ```
 * import * as schema from './schema';
 * const isString: (value: unknown) => value is string = schema.string();
 * ```
 */
function anyString() {
  return function (value) {
    return typeof value === 'string';
  };
}

/**
 * Validate that `value` is a number.
 * NB: this just checks if `typeof value === 'number'`. It will return `true` for `NaN`.
 * NB: this is used as `number` externally via named export.
 * Usage:
 * ```
 * import * as schema from './schema';
 * const isNumber: (value: unknown) => value is number = schema.number();
 * ```
 */
function anyNumber() {
  return function (value) {
    return typeof value === 'number';
  };
}

/**
 * Export `anyString` as `string`. We can't define a function named `string`.
 * Export `anyNumber` as `number`. We can't define a function named `number`.
 */