"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.datetime = datetime;
exports.eventType = eventType;
exports.isTerminatedProcess = isTerminatedProcess;
exports.orderByTime = orderByTime;
exports.processPID = processPID;
exports.processPath = processPath;
exports.userInfoForProcess = userInfoForProcess;
var _ecs_safety_helpers = require("../../../common/endpoint/models/ecs_safety_helpers");
var eventModel = _interopRequireWildcard(require("../../../common/endpoint/models/event"));
var nodeModel = _interopRequireWildcard(require("../../../common/endpoint/models/node"));
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Returns true if the process was terminated.
 */
function isTerminatedProcess(passedEvent) {
  return eventType(passedEvent) === 'processTerminated';
}

/**
 * ms since Unix epoc, based on timestamp.
 * may return NaN if the timestamp wasn't present or was invalid.
 */
function datetime(node) {
  const timestamp = nodeModel.nodeDataTimestamp(node);
  const time = timestamp === undefined ? 0 : new Date(timestamp).getTime();

  // if the date could not be parsed, return null
  return isNaN(time) ? null : time;
}

/**
 * Returns a custom event type for a process event based on the event's metadata.
 */
function eventType(passedEvent) {
  if (eventModel.isLegacyEventSafeVersion(passedEvent)) {
    const {
      endgame: {
        event_type_full: type,
        event_subtype_full: subType
      }
    } = passedEvent;
    if (type === 'process_event') {
      if (subType === 'creation_event' || subType === 'fork_event' || subType === 'exec_event') {
        return 'processCreated';
      } else if (subType === 'already_running') {
        return 'processRan';
      } else if (subType === 'termination_event') {
        return 'processTerminated';
      } else {
        return 'unknownProcessEvent';
      }
    } else if (type === 'alert_event') {
      return 'processCausedAlert';
    }
  } else {
    const type = new Set(eventModel.eventType(passedEvent));
    const category = new Set(eventModel.eventCategory(passedEvent));
    const kind = new Set(eventModel.eventKind(passedEvent));
    if (category.has('process')) {
      // checking for end event.type first as multiple values are possible (merged process events). e.g event.type: ['start', 'end']
      if (type.has('end')) {
        return 'processTerminated';
      } else if (type.has('start') || type.has('change') || type.has('creation')) {
        return 'processCreated';
      } else if (type.has('info')) {
        return 'processRan';
      } else {
        return 'unknownProcessEvent';
      }
    } else if (kind.has('alert')) {
      return 'processCausedAlert';
    }
  }
  return 'unknownEvent';
}

/**
 * Returns the PID for the process on the host
 */
function processPID(event) {
  var _event$process;
  return (0, _ecs_safety_helpers.firstNonNullValue)(eventModel.isLegacyEventSafeVersion(event) ? event.endgame.pid : (_event$process = event.process) === null || _event$process === void 0 ? void 0 : _event$process.pid);
}

/**
 * Returns the process event's path on its host
 */
function processPath(passedEvent) {
  var _passedEvent$process;
  return (0, _ecs_safety_helpers.firstNonNullValue)(eventModel.isLegacyEventSafeVersion(passedEvent) ? passedEvent.endgame.process_path : (_passedEvent$process = passedEvent.process) === null || _passedEvent$process === void 0 ? void 0 : _passedEvent$process.executable);
}

/**
 * Returns the username for the account that ran the process
 */
function userInfoForProcess(passedEvent) {
  return passedEvent.user;
}

/**
 * used to sort events
 */
function orderByTime(first, second) {
  const firstDatetime = datetime(first);
  const secondDatetime = datetime(second);
  if (firstDatetime === secondDatetime) {
    // break ties using an arbitrary (stable) comparison of `nodeID` (which should be unique)
    return String(nodeModel.nodeID(first)).localeCompare(String(nodeModel.nodeID(second)));
  } else if (firstDatetime === null || secondDatetime === null) {
    // sort `null`'s as higher than numbers
    return (firstDatetime === null ? 1 : 0) - (secondDatetime === null ? 1 : 0);
  } else {
    // sort in ascending order.
    return firstDatetime - secondDatetime;
  }
}