"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useFetchFieldValuePairByEventType = exports.EventKind = void 0;
var _esQuery = require("@kbn/es-query");
var _reactQuery = require("@tanstack/react-query");
var _fetch_data = require("../utils/fetch_data");
var _kibana = require("../../../common/lib/kibana");
var _store = require("../../../common/store");
var _use_selector = require("../../../common/hooks/use_selector");
var _use_global_time = require("../../../common/containers/use_global_time");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const QUERY_KEY = 'FetchFieldValuePairByEventType';
let EventKind;
exports.EventKind = EventKind;
(function (EventKind) {
  EventKind["alert"] = "alert";
  EventKind["asset"] = "asset";
  EventKind["enrichment"] = "enrichment";
  EventKind["event"] = "event";
  EventKind["metric"] = "metric";
  EventKind["state"] = "state";
  EventKind["pipeline_error"] = "pipeline_error";
  EventKind["signal"] = "signal";
})(EventKind || (exports.EventKind = EventKind = {}));
/**
 * Hook to retrieve all the unique hosts in the environment that have the field/value pair, using ReactQuery.
 */
const useFetchFieldValuePairByEventType = ({
  highlightedField,
  isActiveTimelines,
  type
}) => {
  const {
    services: {
      data: {
        search: searchService
      }
    }
  } = (0, _kibana.useKibana)();
  const timelineTime = (0, _use_selector.useDeepEqualSelector)(state => _store.inputsSelectors.timelineTimeRangeSelector(state));
  const globalTime = (0, _use_global_time.useGlobalTime)();
  const {
    to,
    from
  } = isActiveTimelines ? timelineTime : globalTime;
  const {
    name,
    values
  } = highlightedField;
  const req = buildSearchRequest(name, values, from, to, type);
  const {
    data,
    isLoading,
    isError
  } = (0, _reactQuery.useQuery)([QUERY_KEY, name, values, from, to, type], () => (0, _fetch_data.createFetchData)(searchService, req), {
    select: res => res.hits.total,
    keepPreviousData: true
  });
  return {
    loading: isLoading,
    error: isError,
    count: data || 0
  };
};

/**
 * Build the search request for the field/values pair, for a date range from/to.
 * We set the size to 0 as we only care about the total number of documents.
 * Passing signalEventKind as true will return only alerts (event.kind === "signal"), otherwise return all other documents (event.kind !== "signal")
 */
exports.useFetchFieldValuePairByEventType = useFetchFieldValuePairByEventType;
const buildSearchRequest = (field, values, from, to, type) => {
  const query = (0, _esQuery.buildEsQuery)(undefined, [], [{
    query: {
      bool: {
        must: [{
          match: {
            [field]: values[0]
          }
        }, {
          range: {
            '@timestamp': {
              gte: from,
              lte: to
            }
          }
        }, ...(type.include ? [{
          match: {
            'event.kind': type.eventKind
          }
        }] : [])],
        ...(type.exclude ? {
          must_not: [{
            match: {
              'event.kind': type.eventKind
            }
          }]
        } : {})
      }
    },
    meta: {}
  }]);
  return {
    params: {
      body: {
        query,
        size: 1000
      }
    }
  };
};