"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.stopDatafeeds = exports.startDatafeeds = exports.setupMlJob = exports.getModules = exports.checkRecognizer = void 0;
var _throw_if_not_ok = require("../ml/api/throw_if_not_ok");
var _kibana = require("../../lib/kibana");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Checks the ML Recognizer API to see if a given indexPattern has any compatible modules
 *
 * @param indexPatternName ES index pattern to check for compatible modules
 * @param signal to cancel request
 *
 * @throws An error if response is not OK
 */
const checkRecognizer = async ({
  indexPatternName,
  signal
}) => _kibana.KibanaServices.get().http.fetch(`/internal/ml/modules/recognize/${indexPatternName}`, {
  method: 'GET',
  version: '1',
  asSystemRequest: true,
  signal
});

/**
 * Returns ML Module for given moduleId. Returns all modules if no moduleId specified
 *
 * @param moduleId id of the module to retrieve
 * @param signal to cancel request
 *
 * @throws An error if response is not OK
 */
exports.checkRecognizer = checkRecognizer;
const getModules = async ({
  moduleId = '',
  signal
}) => _kibana.KibanaServices.get().http.fetch(`/internal/ml/modules/get_module/${moduleId}`, {
  method: 'GET',
  version: '1',
  asSystemRequest: true,
  signal
});

/**
 * Creates ML Jobs + Datafeeds for the given configTemplate + indexPatternName
 *
 * @param configTemplate - name of configTemplate to setup
 * @param indexPatternName - default index pattern configTemplate should be installed with
 * @param jobIdErrorFilter - if provided, filters all errors except for given jobIds
 * @param groups - list of groups to add to jobs being installed
 * @param prefix - prefix to be added to job name
 *
 * @throws An error if response is not OK
 */
exports.getModules = getModules;
const setupMlJob = async ({
  configTemplate,
  indexPatternName = 'auditbeat-*',
  jobIdErrorFilter = [],
  groups = ['security'],
  prefix = ''
}) => {
  const response = await _kibana.KibanaServices.get().http.fetch(`/internal/ml/modules/setup/${configTemplate}`, {
    method: 'POST',
    version: '1',
    body: JSON.stringify({
      prefix,
      groups,
      indexPatternName,
      startDatafeed: false,
      useDedicatedIndex: true,
      applyToAllSpaces: true
    }),
    asSystemRequest: true
  });
  (0, _throw_if_not_ok.throwIfErrorAttachedToSetup)(response, jobIdErrorFilter);
  return response;
};

/**
 * Starts the given dataFeedIds
 *
 * @param datafeedIds
 * @param start
 *
 * @throws An error if response is not OK
 */
exports.setupMlJob = setupMlJob;
const startDatafeeds = async ({
  datafeedIds,
  start = 0
}) => {
  const response = await _kibana.KibanaServices.get().http.fetch('/internal/ml/jobs/force_start_datafeeds', {
    method: 'POST',
    version: '1',
    body: JSON.stringify({
      datafeedIds,
      ...(start !== 0 && {
        start
      })
    }),
    asSystemRequest: true
  });
  (0, _throw_if_not_ok.throwIfErrorAttached)(response, datafeedIds);
  return response;
};

/**
 * Stops the given dataFeedIds and sets the corresponding Job's jobState to closed
 *
 * @param datafeedIds
 *
 * @throws An error if response is not OK
 */
exports.startDatafeeds = startDatafeeds;
const stopDatafeeds = async ({
  datafeedIds
}) => {
  const stopDatafeedsResponse = await _kibana.KibanaServices.get().http.fetch('/internal/ml/jobs/stop_datafeeds', {
    method: 'POST',
    version: '1',
    body: JSON.stringify({
      datafeedIds
    }),
    asSystemRequest: true
  });
  const datafeedPrefix = 'datafeed-';
  const closeJobsResponse = await _kibana.KibanaServices.get().http.fetch('/internal/ml/jobs/close_jobs', {
    method: 'POST',
    version: '1',
    body: JSON.stringify({
      jobIds: datafeedIds.map(dataFeedId => dataFeedId.startsWith(datafeedPrefix) ? dataFeedId.substring(datafeedPrefix.length) : dataFeedId)
    }),
    asSystemRequest: true
  });
  return [stopDatafeedsResponse, closeJobsResponse];
};
exports.stopDatafeeds = stopDatafeeds;