"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.decodeStackTrace = decodeStackTrace;
var _elasticsearch = require("../../common/elasticsearch");
var _profiling = require("../../common/profiling");
var _run_length_encoding = require("../../common/run_length_encoding");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const BASE64_FRAME_ID_LENGTH = 32;
// decodeStackTrace unpacks an encoded stack trace from Elasticsearch
function decodeStackTrace(input) {
  const inputFrameIDs = input.Stacktrace.frame.ids;
  const inputFrameTypes = input.Stacktrace.frame.types;
  const countsFrameIDs = inputFrameIDs.length / BASE64_FRAME_ID_LENGTH;
  const fileIDs = new Array(countsFrameIDs);
  const frameIDs = new Array(countsFrameIDs);
  const addressOrLines = new Array(countsFrameIDs);

  // Step 1: Convert the base64-encoded frameID list into two separate
  // lists (frame IDs and file IDs), both of which are also base64-encoded.
  //
  // To get the frame ID, we grab the next 32 bytes.
  //
  // To get the file ID, we grab the first 22 bytes of the frame ID.
  // However, since the file ID is base64-encoded using 21.33 bytes
  // (16 * 4 / 3), then the 22 bytes have an extra 4 bits from the
  // address (see diagram in definition of EncodedStackTrace).
  for (let i = 0, pos = 0; i < countsFrameIDs; i++, pos += BASE64_FRAME_ID_LENGTH) {
    const frameID = inputFrameIDs.slice(pos, pos + BASE64_FRAME_ID_LENGTH);
    frameIDs[i] = frameID;
    fileIDs[i] = (0, _profiling.getFileIDFromStackFrameID)(frameID);
    addressOrLines[i] = (0, _profiling.getAddressFromStackFrameID)(frameID);
  }

  // Step 2: Convert the run-length byte encoding into a list of uint8s.
  const typeIDs = (0, _run_length_encoding.runLengthDecodeBase64Url)(inputFrameTypes, inputFrameTypes.length, countsFrameIDs);
  return {
    AddressOrLines: addressOrLines,
    FileIDs: fileIDs,
    FrameIDs: frameIDs,
    Types: typeIDs
  };
}