"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerTopNFunctionsSearchRoute = registerTopNFunctionsSearchRoute;
var _configSchema = require("@kbn/config-schema");
var _common = require("../../common");
var _functions = require("../../common/functions");
var _handle_route_error_handler = require("../utils/handle_route_error_handler");
var _with_profiling_span = require("../utils/with_profiling_span");
var _compat = require("./compat");
var _query = require("./query");
var _search_stacktraces = require("./search_stacktraces");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const querySchema = _configSchema.schema.object({
  timeFrom: _configSchema.schema.number(),
  timeTo: _configSchema.schema.number(),
  startIndex: _configSchema.schema.number(),
  endIndex: _configSchema.schema.number(),
  kuery: _configSchema.schema.string()
});
function registerTopNFunctionsSearchRoute({
  router,
  logger,
  services: {
    createProfilingEsClient
  }
}) {
  const paths = (0, _common.getRoutePaths)();
  router.get({
    path: paths.TopNFunctions,
    options: {
      tags: ['access:profiling']
    },
    validate: {
      query: querySchema
    }
  }, async (context, request, response) => {
    try {
      const {
        timeFrom,
        timeTo,
        startIndex,
        endIndex,
        kuery
      } = request.query;
      const targetSampleSize = 20000; // minimum number of samples to get statistically sound results
      const esClient = await (0, _compat.getClient)(context);
      const profilingElasticsearchClient = createProfilingEsClient({
        request,
        esClient
      });
      const filter = (0, _query.createCommonFilter)({
        timeFrom,
        timeTo,
        kuery
      });
      const {
        stackTraceEvents,
        stackTraces,
        executables,
        stackFrames,
        samplingRate
      } = await (0, _search_stacktraces.searchStackTraces)({
        client: profilingElasticsearchClient,
        filter,
        sampleSize: targetSampleSize
      });
      const topNFunctions = await (0, _with_profiling_span.withProfilingSpan)('create_topn_functions', async () => {
        return (0, _functions.createTopNFunctions)(stackTraceEvents, stackTraces, stackFrames, executables, startIndex, endIndex, samplingRate);
      });
      return response.ok({
        body: topNFunctions
      });
    } catch (error) {
      return (0, _handle_route_error_handler.handleRouteHandlerError)({
        error,
        logger,
        response,
        message: 'Error while fetching TopN functions'
      });
    }
  });
}