"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ApmTransactionErrorRateTransformGenerator = void 0;
var _sloSchema = require("@kbn/slo-schema");
var _errors = require("../../../errors");
var _slo_transform_template = require("../../../assets/transform_templates/slo_transform_template");
var _ = require(".");
var _constants = require("../../../assets/constants");
var _common = require("./common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class ApmTransactionErrorRateTransformGenerator extends _.TransformGenerator {
  getTransformParams(slo) {
    if (!_sloSchema.apmTransactionErrorRateIndicatorSchema.is(slo.indicator)) {
      throw new _errors.InvalidTransformError(`Cannot handle SLO of indicator type: ${slo.indicator.type}`);
    }
    return (0, _slo_transform_template.getSLOTransformTemplate)(this.buildTransformId(slo), this.buildDescription(slo), this.buildSource(slo, slo.indicator), this.buildDestination(), this.buildGroupBy(slo), this.buildAggregations(slo), this.buildSettings(slo));
  }
  buildTransformId(slo) {
    return (0, _constants.getSLOTransformId)(slo.id, slo.revision);
  }
  buildSource(slo, indicator) {
    const queryFilter = [{
      range: {
        '@timestamp': {
          gte: `now-${slo.timeWindow.duration.format()}`
        }
      }
    }];
    if (indicator.params.service !== _sloSchema.ALL_VALUE) {
      queryFilter.push({
        match: {
          'service.name': indicator.params.service
        }
      });
    }
    if (indicator.params.environment !== _sloSchema.ALL_VALUE) {
      queryFilter.push({
        match: {
          'service.environment': indicator.params.environment
        }
      });
    }
    if (indicator.params.transactionName !== _sloSchema.ALL_VALUE) {
      queryFilter.push({
        match: {
          'transaction.name': indicator.params.transactionName
        }
      });
    }
    if (indicator.params.transactionType !== _sloSchema.ALL_VALUE) {
      queryFilter.push({
        match: {
          'transaction.type': indicator.params.transactionType
        }
      });
    }
    if (indicator.params.filter) {
      queryFilter.push((0, _.getElastichsearchQueryOrThrow)(indicator.params.filter));
    }
    return {
      index: (0, _common.parseIndex)(indicator.params.index),
      runtime_mappings: this.buildCommonRuntimeMappings(slo),
      query: {
        bool: {
          filter: [{
            terms: {
              'processor.event': ['metric']
            }
          }, {
            term: {
              'metricset.name': 'transaction'
            }
          }, {
            exists: {
              field: 'transaction.duration.histogram'
            }
          }, {
            exists: {
              field: 'transaction.result'
            }
          }, ...queryFilter]
        }
      }
    };
  }
  buildDestination() {
    return {
      pipeline: _constants.SLO_INGEST_PIPELINE_NAME,
      index: _constants.SLO_DESTINATION_INDEX_NAME
    };
  }
  buildAggregations(slo) {
    return {
      'slo.numerator': {
        filter: {
          bool: {
            should: {
              match: {
                'event.outcome': 'success'
              }
            }
          }
        }
      },
      'slo.denominator': {
        value_count: {
          field: 'transaction.duration.histogram'
        }
      },
      ...(_sloSchema.timeslicesBudgetingMethodSchema.is(slo.budgetingMethod) && {
        'slo.isGoodSlice': {
          bucket_script: {
            buckets_path: {
              goodEvents: 'slo.numerator>_count',
              totalEvents: 'slo.denominator.value'
            },
            script: `params.goodEvents / params.totalEvents >= ${slo.objective.timesliceTarget} ? 1 : 0`
          }
        }
      })
    };
  }
}
exports.ApmTransactionErrorRateTransformGenerator = ApmTransactionErrorRateTransformGenerator;