"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DefaultSummaryClient = void 0;
var _sloSchema = require("@kbn/slo-schema");
var _moment = _interopRequireDefault(require("moment"));
var _constants = require("../../assets/constants");
var _services = require("../../domain/services");
var _date_range = require("../../domain/services/date_range");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class DefaultSummaryClient {
  constructor(esClient) {
    this.esClient = esClient;
  }
  async fetchSummary(sloList) {
    const dateRangeBySlo = sloList.reduce((acc, slo) => {
      acc[slo.id] = (0, _date_range.toDateRange)(slo.timeWindow);
      return acc;
    }, {});
    const searches = sloList.flatMap(slo => [{
      index: _constants.SLO_DESTINATION_INDEX_PATTERN
    }, generateSearchQuery(slo, dateRangeBySlo[slo.id])]);
    const summaryBySlo = {};
    if (searches.length === 0) {
      return summaryBySlo;
    }
    const result = await this.esClient.msearch({
      searches
    });
    for (let i = 0; i < result.responses.length; i++) {
      var _aggregations$good$va, _aggregations$good, _aggregations$total$v, _aggregations$total;
      const slo = sloList[i];

      // @ts-ignore
      const {
        aggregations = {}
      } = result.responses[i];
      const good = (_aggregations$good$va = aggregations === null || aggregations === void 0 ? void 0 : (_aggregations$good = aggregations.good) === null || _aggregations$good === void 0 ? void 0 : _aggregations$good.value) !== null && _aggregations$good$va !== void 0 ? _aggregations$good$va : 0;
      const total = (_aggregations$total$v = aggregations === null || aggregations === void 0 ? void 0 : (_aggregations$total = aggregations.total) === null || _aggregations$total === void 0 ? void 0 : _aggregations$total.value) !== null && _aggregations$total$v !== void 0 ? _aggregations$total$v : 0;
      const sliValue = (0, _services.computeSLI)(good, total);
      const initialErrorBudget = 1 - slo.objective.target;
      let errorBudget;
      if (_sloSchema.calendarAlignedTimeWindowSchema.is(slo.timeWindow) && _sloSchema.timeslicesBudgetingMethodSchema.is(slo.budgetingMethod)) {
        const totalSlices = computeTotalSlicesFromDateRange(dateRangeBySlo[slo.id], slo.objective.timesliceWindow);
        const consumedErrorBudget = sliValue < 0 ? 0 : (total - good) / (totalSlices * initialErrorBudget);
        errorBudget = (0, _services.toErrorBudget)(initialErrorBudget, consumedErrorBudget);
      } else {
        const consumedErrorBudget = sliValue < 0 ? 0 : (1 - sliValue) / initialErrorBudget;
        errorBudget = (0, _services.toErrorBudget)(initialErrorBudget, consumedErrorBudget, _sloSchema.calendarAlignedTimeWindowSchema.is(slo.timeWindow));
      }
      summaryBySlo[slo.id] = {
        sliValue,
        errorBudget,
        status: (0, _services.computeSummaryStatus)(slo, sliValue, errorBudget)
      };
    }
    return summaryBySlo;
  }
}
exports.DefaultSummaryClient = DefaultSummaryClient;
function computeTotalSlicesFromDateRange(dateRange, timesliceWindow) {
  const dateRangeDurationInUnit = (0, _moment.default)(dateRange.to).diff(dateRange.from, (0, _sloSchema.toMomentUnitOfTime)(timesliceWindow.unit));
  return Math.ceil(dateRangeDurationInUnit / timesliceWindow.value);
}
function generateSearchQuery(slo, dateRange) {
  return {
    size: 0,
    query: {
      bool: {
        filter: [{
          term: {
            'slo.id': slo.id
          }
        }, {
          term: {
            'slo.revision': slo.revision
          }
        }, {
          range: {
            '@timestamp': {
              gte: dateRange.from.toISOString(),
              lt: dateRange.to.toISOString()
            }
          }
        }]
      }
    },
    ...(_sloSchema.occurrencesBudgetingMethodSchema.is(slo.budgetingMethod) && {
      aggs: {
        good: {
          sum: {
            field: 'slo.numerator'
          }
        },
        total: {
          sum: {
            field: 'slo.denominator'
          }
        }
      }
    }),
    ...(_sloSchema.timeslicesBudgetingMethodSchema.is(slo.budgetingMethod) && {
      aggs: {
        good: {
          sum: {
            field: 'slo.isGoodSlice'
          }
        },
        total: {
          value_count: {
            field: 'slo.isGoodSlice'
          }
        }
      }
    })
  };
}