"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SortField = exports.SortDirection = exports.KibanaSavedObjectsCompositeSLORepository = void 0;
var _sloSchema = require("@kbn/slo-schema");
var _Either = require("fp-ts/lib/Either");
var _pipeable = require("fp-ts/lib/pipeable");
var t = _interopRequireWildcard(require("io-ts"));
var _errors = require("../../errors");
var _saved_objects = require("../../saved_objects");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SortDirection = {
  Asc: 'Asc',
  Desc: 'Desc'
};
exports.SortDirection = SortDirection;
const SortField = {
  CreationTime: 'CreationTime',
  IndicatorType: 'IndicatorType'
};
exports.SortField = SortField;
const SAVED_OBJECT_ATTRIBUTES_PATH = 'composite-slo.attributes';
class KibanaSavedObjectsCompositeSLORepository {
  constructor(soClient) {
    this.soClient = soClient;
  }
  async save(compositeSlo, options = {
    throwOnConflict: false
  }) {
    let existingSavedObjectId;
    const findResponse = await this.soClient.find({
      type: _saved_objects.SO_COMPOSITE_SLO_TYPE,
      page: 1,
      perPage: 1,
      filter: `${SAVED_OBJECT_ATTRIBUTES_PATH}.id:(${compositeSlo.id})`
    });
    if (findResponse.total === 1) {
      if (options.throwOnConflict) {
        throw new _errors.CompositeSLOIdConflict(`Composite SLO [${compositeSlo.id}] already exists`);
      }
      existingSavedObjectId = findResponse.saved_objects[0].id;
    }
    const createResponse = await this.soClient.create(_saved_objects.SO_COMPOSITE_SLO_TYPE, toStoredCompositeSLO(compositeSlo), {
      id: existingSavedObjectId,
      overwrite: true
    });
    return toCompositeSLO(createResponse.attributes);
  }
  async deleteById(id) {
    const response = await this.soClient.find({
      type: _saved_objects.SO_COMPOSITE_SLO_TYPE,
      page: 1,
      perPage: 1,
      filter: `${SAVED_OBJECT_ATTRIBUTES_PATH}.id:(${id})`
    });
    if (response.total === 0) {
      throw new _errors.CompositeSLONotFound(`Composite SLO [${id}] not found`);
    }
    await this.soClient.delete(_saved_objects.SO_COMPOSITE_SLO_TYPE, response.saved_objects[0].id);
  }
  async findById(id) {
    const response = await this.soClient.find({
      type: _saved_objects.SO_COMPOSITE_SLO_TYPE,
      page: 1,
      perPage: 1,
      filter: `${SAVED_OBJECT_ATTRIBUTES_PATH}.id:(${id})`
    });
    if (response.total === 0) {
      throw new _errors.CompositeSLONotFound(`Composite SLO [${id}] not found`);
    }
    return toCompositeSLO(response.saved_objects[0].attributes);
  }
  async find(criteria, sort, pagination) {
    const {
      search,
      searchFields
    } = buildSearch(criteria);
    const {
      sortField,
      sortOrder
    } = buildSortQuery(sort);
    const response = await this.soClient.find({
      type: _saved_objects.SO_COMPOSITE_SLO_TYPE,
      page: pagination.page,
      perPage: pagination.perPage,
      search,
      searchFields,
      sortField,
      sortOrder
    });
    return {
      total: response.total,
      page: response.page,
      perPage: response.per_page,
      results: response.saved_objects.map(so => toCompositeSLO(so.attributes))
    };
  }
}
exports.KibanaSavedObjectsCompositeSLORepository = KibanaSavedObjectsCompositeSLORepository;
function toStoredCompositeSLO(compositeSlo) {
  return _sloSchema.compositeSloSchema.encode(compositeSlo);
}
function toCompositeSLO(storedCompositeSlo) {
  return (0, _pipeable.pipe)(_sloSchema.compositeSloSchema.decode(storedCompositeSlo), (0, _Either.fold)(() => {
    throw new Error(`Invalid stored composite SLO [${storedCompositeSlo.id}]`);
  }, t.identity));
}
function buildSearch(criteria) {
  if (!criteria.name) {
    return {
      search: undefined,
      searchFields: undefined
    };
  }
  return {
    search: addWildcardsIfAbsent(criteria.name),
    searchFields: ['name']
  };
}
function buildSortQuery(sort) {
  let sortField;
  switch (sort.field) {
    case SortField.CreationTime:
    default:
      sortField = 'created_at';
      break;
  }
  return {
    sortField,
    sortOrder: sort.direction === SortDirection.Desc ? 'desc' : 'asc'
  };
}
const WILDCARD_CHAR = '*';
function addWildcardsIfAbsent(value) {
  let updatedValue = value;
  if (updatedValue.substring(0, 1) !== WILDCARD_CHAR) {
    updatedValue = `${WILDCARD_CHAR}${updatedValue}`;
  }
  if (value.substring(value.length - 1) !== WILDCARD_CHAR) {
    updatedValue = `${updatedValue}${WILDCARD_CHAR}`;
  }
  return updatedValue;
}