"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AutocompleteListener = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _dymanic_item = require("./dymanic_item");
var _esql_parser = require("../../antlr/esql_parser");
var _autocomplete_definitions = require("./autocomplete_definitions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

class AutocompleteListener {
  constructor() {
    (0, _defineProperty2.default)(this, "suggestions", []);
    (0, _defineProperty2.default)(this, "userDefinedVariables", {
      sourceIdentifiers: []
    });
    (0, _defineProperty2.default)(this, "tables", []);
    (0, _defineProperty2.default)(this, "parentContext", void 0);
  }
  get fields() {
    return this.tables.length > 1 ? (0, _autocomplete_definitions.buildConstantsDefinitions)(this.tables.at(-2)) : [_dymanic_item.DynamicAutocompleteItem.FieldIdentifier];
  }
  get hasSuggestions() {
    return Boolean(this.suggestions.length);
  }
  isTerminalNodeExists(node) {
    var _node$payload;
    return node && ((_node$payload = node.payload) === null || _node$payload === void 0 ? void 0 : _node$payload.startIndex) >= 0;
  }
  getEndCommandSuggestions(skipDefinitions = []) {
    const suggestions = [_autocomplete_definitions.pipeDefinition];
    if (!skipDefinitions.find(i => i === _autocomplete_definitions.byOperatorDefinition) && this.parentContext === _esql_parser.esql_parser.STATS) {
      suggestions.push(_autocomplete_definitions.byOperatorDefinition);
    }
    return suggestions;
  }
  getNewVarName() {
    const vars = this.tables.flat();
    let index = 0;
    while (true) {
      const value = `var${index}`;
      if (!vars.includes(value)) {
        return value;
      }
      index++;
    }
  }
  getAutocompleteSuggestions() {
    return {
      suggestions: this.suggestions,
      userDefinedVariables: this.userDefinedVariables
    };
  }

  /** ESQLParserListener fields  **/

  enterSourceCommand(ctx) {
    this.suggestions = [];
  }
  exitSourceCommand(ctx) {
    if (ctx.exception) {
      this.suggestions = _autocomplete_definitions.sourceCommandsDefinitions;
    } else if (!this.hasSuggestions) {
      this.suggestions = this.getEndCommandSuggestions();
    }
  }
  exitSourceIdentifier(ctx) {
    if (!ctx.childCount) {
      this.suggestions = [_dymanic_item.DynamicAutocompleteItem.SourceIdentifier];
    } else if (!ctx.exception && ctx.text) {
      this.userDefinedVariables.sourceIdentifiers.push(ctx.text);
    }
  }
  enterProcessingCommand(ctx) {
    this.tables.push([]);
    this.suggestions = [];
    this.parentContext = undefined;
  }
  exitProcessingCommand(ctx) {
    if (ctx.exception) {
      this.suggestions = _autocomplete_definitions.processingCommandsDefinitions;
    }
    this.parentContext = undefined;
  }
  enterStatsCommand(ctx) {
    this.suggestions = [];
    this.parentContext = _esql_parser.esql_parser.STATS;
  }
  enterEvalCommand(ctx) {
    this.suggestions = [];
    this.parentContext = _esql_parser.esql_parser.EVAL;
  }
  exitStatsCommand(ctx) {
    const qn = ctx.qualifiedNames();
    if (qn && qn.text) {
      this.suggestions = this.getEndCommandSuggestions([_autocomplete_definitions.byOperatorDefinition]);
    }
  }
  exitQualifiedName(ctx) {
    if (ctx.identifier().some(i => !(this.isTerminalNodeExists(i.QUOTED_IDENTIFIER()) || this.isTerminalNodeExists(i.UNQUOTED_IDENTIFIER())))) {
      this.suggestions = this.fields;
    }
  }
  enterField(ctx) {
    this.suggestions = [];
  }
  exitField(ctx) {
    const hasAssign = this.isTerminalNodeExists(ctx.ASSIGN());
    if (ctx.exception) {
      if (!hasAssign) {
        this.suggestions = [(0, _autocomplete_definitions.buildNewVarDefinition)(this.getNewVarName())];
        return;
      }
    } else {
      if (!hasAssign) {
        this.suggestions = [_autocomplete_definitions.assignOperatorDefinition];
      }
    }
  }
  exitUserVariable(ctx) {
    if (!ctx.exception && ctx.text) {
      var _this$tables$at;
      (_this$tables$at = this.tables.at(-1)) === null || _this$tables$at === void 0 ? void 0 : _this$tables$at.push(ctx.text);
    }
  }
  enterBooleanExpression(ctx) {
    this.suggestions = [];
  }
  exitBooleanExpression(ctx) {
    if (ctx.exception) {
      const ve = ctx.valueExpression();
      if (!ve) {
        if (this.parentContext === _esql_parser.esql_parser.STATS) {
          this.suggestions = [..._autocomplete_definitions.aggregationFunctionsDefinitions, ...this.fields];
          return;
        }
        if (this.parentContext === _esql_parser.esql_parser.EVAL) {
          this.suggestions = [_autocomplete_definitions.roundCommandDefinition, ...this.fields];
          return;
        }
      }
    }
  }
  exitValueExpression(ctx) {
    const isInStats = this.parentContext === _esql_parser.esql_parser.STATS;
    const isInEval = this.parentContext === _esql_parser.esql_parser.EVAL;
    if (this.parentContext && (isInStats || isInEval)) {
      const hasFN = ctx.tryGetToken(_esql_parser.esql_parser.UNARY_FUNCTION, 0);
      const hasLP = ctx.tryGetToken(_esql_parser.esql_parser.LP, 0);
      const hasRP = ctx.tryGetToken(_esql_parser.esql_parser.RP, 0);
      if (hasFN) {
        if (!hasLP) {
          this.suggestions = [_autocomplete_definitions.openBracketDefinition];
          return;
        }
        if (!hasRP) {
          if (ctx.childCount === 3) {
            this.suggestions = [_autocomplete_definitions.closeBracketDefinition, ...this.fields];
            return;
          }
        }
      } else {
        if (ctx.childCount === 1) {
          this.suggestions = [...this.getEndCommandSuggestions(), ...(isInEval ? _autocomplete_definitions.mathOperatorsCommandsDefinitions : [])];
          return;
        }
      }
      this.suggestions = this.fields;
    }
  }
  enterWhereCommand(ctx) {
    this.suggestions = [];
    this.parentContext = _esql_parser.esql_parser.WHERE;
  }
  exitWhereCommand(ctx) {
    const booleanExpression = ctx.booleanExpression();
    if (booleanExpression.exception) {
      this.suggestions = this.fields;
      return;
    } else {
      const innerBooleanExpressions = booleanExpression.getRuleContexts(_esql_parser.BooleanExpressionContext);
      if (innerBooleanExpressions.some(be => be.exception)) {
        this.suggestions = this.fields;
        return;
      }
    }
    if (!this.hasSuggestions) {
      this.suggestions = _autocomplete_definitions.comparisonCommandsDefinitions;
    }
  }
  exitComparison(ctx) {
    const operatorExpression = ctx.operatorExpression();
    if (operatorExpression.some(o => o.exception)) {
      this.suggestions = this.fields;
      return;
    }
    this.suggestions = [..._autocomplete_definitions.comparisonOperatorsCommandsDefinitions, ...this.getEndCommandSuggestions()];
  }
  exitOrderExpression(ctx) {
    if (ctx.booleanExpression().exception) {
      this.suggestions = this.fields;
      return;
    }
    if (!this.isTerminalNodeExists(ctx.ORDERING())) {
      this.suggestions = _autocomplete_definitions.orderingCommandsDefinitions;
      return;
    }
    if (!this.isTerminalNodeExists(ctx.NULLS_ORDERING())) {
      this.suggestions = [_autocomplete_definitions.nullsCommandsDefinition];
      return;
    }
    if (!this.isTerminalNodeExists(ctx.NULLS_ORDERING_DIRECTION())) {
      this.suggestions = _autocomplete_definitions.nullsOrderingCommandsDefinitions;
      return;
    }
  }
  exitLimitCommand(ctx) {
    const DEFAULT_LIMIT_SIZE = 1000;
    if (!this.isTerminalNodeExists(ctx.INTEGER_LITERAL())) {
      this.suggestions = (0, _autocomplete_definitions.buildConstantsDefinitions)([DEFAULT_LIMIT_SIZE.toString()], '');
    } else {
      this.suggestions = this.getEndCommandSuggestions();
    }
  }
}
exports.AutocompleteListener = AutocompleteListener;