"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDataViewAndSavedSearchCallback = void 0;
exports.getDataViewById = getDataViewById;
exports.getDataViewIdFromName = getDataViewIdFromName;
exports.getDataViewNames = getDataViewNames;
exports.getQueryFromSavedSearchObject = getQueryFromSavedSearchObject;
exports.isCcsIndexPattern = isCcsIndexPattern;
exports.timeBasedIndexCheck = timeBasedIndexCheck;
var _i18n = require("@kbn/i18n");
var _dependency_cache = require("./dependency_cache");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getDataViewNames() {
  const dataViewsService = (0, _dependency_cache.getDataViews)();
  if (dataViewsService === null) {
    throw new Error('Data views are not initialized!');
  }
  return (await dataViewsService.getIdsWithTitle()).map(({
    title
  }) => title);
}
async function getDataViewIdFromName(name) {
  var _dataView$id;
  const dataViewsService = (0, _dependency_cache.getDataViews)();
  if (dataViewsService === null) {
    throw new Error('Data views are not initialized!');
  }
  const dataViews = await dataViewsService.find(name);
  const dataView = dataViews.find(dv => dv.getIndexPattern() === name);
  if (!dataView) {
    return null;
  }
  return (_dataView$id = dataView.id) !== null && _dataView$id !== void 0 ? _dataView$id : dataView.getIndexPattern();
}
function getDataViewById(id) {
  const dataViewsService = (0, _dependency_cache.getDataViews)();
  if (dataViewsService === null) {
    throw new Error('Data views are not initialized!');
  }
  if (id) {
    return dataViewsService.get(id);
  } else {
    return dataViewsService.create({});
  }
}
const getDataViewAndSavedSearchCallback = deps => async savedSearchId => {
  var _ss$references, _ss$references$find;
  const resp = {
    savedSearch: null,
    dataView: null
  };
  if (savedSearchId === undefined) {
    return resp;
  }
  const ss = await deps.savedSearchService.get(savedSearchId);
  if (ss === null) {
    return resp;
  }
  const dataViewId = (_ss$references = ss.references) === null || _ss$references === void 0 ? void 0 : (_ss$references$find = _ss$references.find(r => r.type === 'index-pattern')) === null || _ss$references$find === void 0 ? void 0 : _ss$references$find.id;
  resp.dataView = await deps.dataViewsService.get(dataViewId);
  resp.savedSearch = ss;
  return resp;
};
exports.getDataViewAndSavedSearchCallback = getDataViewAndSavedSearchCallback;
function getQueryFromSavedSearchObject(savedSearch) {
  return {
    query: savedSearch.searchSource.getField('query'),
    filter: savedSearch.searchSource.getField('filter')
  };
}

/**
 * Returns true if the index passed in is time based
 * an optional flag will trigger the display a notification at the top of the page
 * warning that the index is not time based
 */
function timeBasedIndexCheck(dataView, showNotification = false) {
  if (!dataView.isTimeBased()) {
    if (showNotification) {
      const toastNotifications = (0, _dependency_cache.getToastNotifications)();
      toastNotifications.addWarning({
        title: _i18n.i18n.translate('xpack.ml.dataViewNotBasedOnTimeSeriesNotificationTitle', {
          defaultMessage: 'The data view {dataViewIndexPattern} is not based on a time series',
          values: {
            dataViewIndexPattern: dataView.getIndexPattern()
          }
        }),
        text: _i18n.i18n.translate('xpack.ml.dataViewNotBasedOnTimeSeriesNotificationDescription', {
          defaultMessage: 'Anomaly detection only runs over time-based indices'
        })
      });
    }
    return false;
  } else {
    return true;
  }
}

/**
 * Returns true if the data view index pattern contains a :
 * which means it is cross-cluster
 */
function isCcsIndexPattern(dataViewIndexPattern) {
  return dataViewIndexPattern.includes(':');
}