"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getMapAttributeService = getMapAttributeService;
var _constants = require("../common/constants");
var _kibana_services = require("./kibana_services");
var _references = require("../common/migrations/references");
var _content_management = require("./content_management");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

let mapAttributeService = null;
function getMapAttributeService() {
  if (mapAttributeService) {
    return mapAttributeService;
  }
  mapAttributeService = (0, _kibana_services.getEmbeddableService)().getAttributeService(_constants.MAP_SAVED_OBJECT_TYPE, {
    saveMethod: async (attributes, savedObjectId) => {
      // AttributeService "attributes" contains "references" as a child.
      // SavedObjectClient "attributes" uses "references" as a sibling.
      // https://github.com/elastic/kibana/issues/83133
      const savedObjectClientReferences = attributes.references;
      const savedObjectClientAttributes = {
        ...attributes
      };
      delete savedObjectClientAttributes.references;
      const {
        attributes: updatedAttributes,
        references
      } = (0, _references.extractReferences)({
        attributes: savedObjectClientAttributes,
        references: savedObjectClientReferences
      });
      const {
        item: {
          id
        }
      } = await (savedObjectId ? _content_management.mapsClient.update({
        id: savedObjectId,
        data: updatedAttributes,
        options: {
          references
        }
      }) : _content_management.mapsClient.create({
        data: updatedAttributes,
        options: {
          references
        }
      }));
      return {
        id
      };
    },
    unwrapMethod: async savedObjectId => {
      const {
        item: savedObject,
        meta: {
          outcome,
          aliasPurpose,
          aliasTargetId
        }
      } = await _content_management.mapsClient.get(savedObjectId);
      if (savedObject.error) {
        throw savedObject.error;
      }
      const {
        attributes
      } = (0, _references.injectReferences)(savedObject);
      return {
        attributes: {
          ...attributes,
          references: savedObject.references
        },
        metaInfo: {
          sharingSavedObjectProps: {
            aliasTargetId,
            outcome,
            aliasPurpose,
            sourceId: savedObjectId
          }
        }
      };
    },
    checkForDuplicateTitle: props => {
      return (0, _content_management.checkForDuplicateTitle)({
        title: props.newTitle,
        copyOnSave: false,
        lastSavedTitle: '',
        isTitleDuplicateConfirmed: props.isTitleDuplicateConfirmed,
        getDisplayName: () => _constants.MAP_EMBEDDABLE_NAME,
        onTitleDuplicate: props.onTitleDuplicate
      }, {
        overlays: (0, _kibana_services.getCoreOverlays)()
      });
    }
  });
  return mapAttributeService;
}