"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.formulaOperation = void 0;
exports.isFormulaIndexPatternColumn = isFormulaIndexPatternColumn;
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _utils = require("../../../../../utils");
var _validation = require("./validation");
var _editor = require("./editor");
var _parse = require("./parse");
var _generate = require("./generate");
var _util = require("./util");
var _layer_helpers = require("../../layer_helpers");
var _helpers = require("../helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const defaultLabel = _i18n.i18n.translate('xpack.lens.indexPattern.formulaLabel', {
  defaultMessage: 'Formula'
});
function isFormulaIndexPatternColumn(column) {
  return 'params' in column && 'formula' in column.params;
}
const formulaOperation = {
  type: 'formula',
  displayName: defaultLabel,
  getDefaultLabel: (column, indexPattern) => {
    var _column$params$formul;
    return (_column$params$formul = column.params.formula) !== null && _column$params$formul !== void 0 ? _column$params$formul : defaultLabel;
  },
  input: 'managedReference',
  hidden: true,
  filterable: {
    helpMessage: _i18n.i18n.translate('xpack.lens.indexPattern.formulaFilterableHelpText', {
      defaultMessage: 'The provided filter will be applied to the entire formula.'
    })
  },
  canReduceTimeRange: {
    helpMessage: _i18n.i18n.translate('xpack.lens.indexPattern.formulaCanReduceTimeRangeHelpText', {
      defaultMessage: 'Applies to the entire formula.'
    })
  },
  getDisabledStatus(indexPattern) {
    return undefined;
  },
  getErrorMessage(layer, columnId, indexPattern, dateRange, operationDefinitionMap, targetBars) {
    const column = layer.columns[columnId];
    if (!column.params.formula || !operationDefinitionMap) {
      return;
    }
    const visibleOperationsMap = (0, _util.filterByVisibleOperation)(operationDefinitionMap);
    const {
      root,
      error
    } = (0, _validation.tryToParse)(column.params.formula, visibleOperationsMap);
    if (error || root == null) {
      return error !== null && error !== void 0 && error.message ? [error.message] : [];
    }
    const errors = (0, _validation.runASTValidation)(root, layer, indexPattern, visibleOperationsMap, column, dateRange);
    if (errors.length) {
      // remove duplicates
      return (0, _lodash.uniqBy)(errors, ({
        message
      }) => message).map(({
        type,
        message,
        extraInfo
      }) => type === 'missingField' && extraInfo !== null && extraInfo !== void 0 && extraInfo.missingFields ? (0, _helpers.generateMissingFieldMessage)(extraInfo.missingFields, columnId) : message);
    }
    const managedColumns = (0, _layer_helpers.getManagedColumnsFrom)(columnId, layer.columns);
    const innerErrors = [...managedColumns.flatMap(([id, col]) => {
      const def = visibleOperationsMap[col.operationType];
      if (def !== null && def !== void 0 && def.getErrorMessage) {
        // TOOD: it would be nice to have nicer column names here rather than `Part of <formula content>`
        const messages = def.getErrorMessage(layer, id, indexPattern, dateRange, visibleOperationsMap, targetBars);
        return messages || [];
      }
      return [];
    }).filter(_utils.nonNullable)
    // dedup messages with the same content
    .reduce((memo, message) => {
      memo.add(message);
      return memo;
    }, new Set())];
    const hasBuckets = layer.columnOrder.some(colId => layer.columns[colId].isBucketed);
    const hasOtherMetrics = layer.columnOrder.some(colId => {
      const col = layer.columns[colId];
      return !col.isBucketed && !col.isStaticValue && col.operationType !== 'math' && col.operationType !== 'formula';
    });
    // What happens when it transition from an error state to a new valid state?
    // the "hasOtherMetrics" might be false as the formula hasn't had time to
    // populate all the referenced columns yet. So check if there are managedColumns
    // (if no error is present, there's at least one other math column present)
    const hasBeenEvaluated = !errors.length && managedColumns.length;
    if (hasBuckets && !hasOtherMetrics && hasBeenEvaluated) {
      innerErrors.push({
        message: _i18n.i18n.translate('xpack.lens.indexPattern.noRealMetricError', {
          defaultMessage: 'A layer with only static values will not show results, use at least one dynamic metric'
        })
      });
    }
    return innerErrors.length ? innerErrors : undefined;
  },
  getPossibleOperation() {
    return {
      dataType: 'number',
      isBucketed: false,
      scale: 'ratio'
    };
  },
  toExpression: (layer, columnId) => {
    var _params$formula;
    const currentColumn = layer.columns[columnId];
    const params = currentColumn.params;
    // TODO: improve this logic
    const useDisplayLabel = currentColumn.label !== defaultLabel;
    const label = !(params !== null && params !== void 0 && params.isFormulaBroken) ? useDisplayLabel ? currentColumn.label : (_params$formula = params === null || params === void 0 ? void 0 : params.formula) !== null && _params$formula !== void 0 ? _params$formula : defaultLabel : defaultLabel;
    return [{
      type: 'function',
      function: currentColumn.references.length ? 'mathColumn' : 'mapColumn',
      arguments: {
        id: [columnId],
        name: [label || defaultLabel],
        ...(currentColumn.references.length ? {
          castColumns: [currentColumn.references[0]]
        } : {}),
        expression: [currentColumn.references.length ? `"${currentColumn.references[0]}"` : '']
      }
    }];
  },
  buildColumn({
    previousColumn,
    layer,
    indexPattern
  }, columnParams, operationDefinitionMap) {
    let previousFormula = '';
    if (previousColumn) {
      previousFormula = (0, _generate.generateFormula)(previousColumn, layer, previousFormula, operationDefinitionMap);
    }
    // carry over the format settings from previous operation for seamless transfer
    // NOTE: this works only for non-default formatters set in Lens
    let format = {};
    if (previousColumn && (0, _helpers.isColumnFormatted)(previousColumn)) {
      var _previousColumn$param;
      format = {
        format: (_previousColumn$param = previousColumn.params) === null || _previousColumn$param === void 0 ? void 0 : _previousColumn$param.format
      };
    }
    if (columnParams !== null && columnParams !== void 0 && columnParams.format) {
      format = {
        format: columnParams.format
      };
    }
    const isPreviousFormulaColumn = (previousColumn === null || previousColumn === void 0 ? void 0 : previousColumn.operationType) === 'formula';
    return {
      label: previousFormula || defaultLabel,
      dataType: 'number',
      operationType: 'formula',
      isBucketed: false,
      scale: 'ratio',
      params: previousFormula ? {
        formula: previousFormula,
        isFormulaBroken: false,
        ...format,
        ...(columnParams !== null && columnParams !== void 0 && columnParams.formula ? {
          formula: columnParams === null || columnParams === void 0 ? void 0 : columnParams.formula
        } : {})
      } : {
        ...format,
        ...(columnParams !== null && columnParams !== void 0 && columnParams.formula ? {
          formula: columnParams === null || columnParams === void 0 ? void 0 : columnParams.formula
        } : {})
      },
      references: [],
      // carry over the filter if coming from another formula,
      // otherwise the filter has been already migrated into the formula text
      filter: isPreviousFormulaColumn ? (0, _helpers.getFilter)(previousColumn, columnParams) : undefined,
      reducedTimeRange: isPreviousFormulaColumn ? previousColumn.reducedTimeRange : undefined,
      timeScale: previousColumn === null || previousColumn === void 0 ? void 0 : previousColumn.timeScale
    };
  },
  isTransferable: () => {
    return true;
  },
  createCopy(layers, source, target, operationDefinitionMap) {
    const currentColumn = layers[source.layerId].columns[source.columnId];
    const modifiedLayer = (0, _parse.insertOrReplaceFormulaColumn)(target.columnId, currentColumn, layers[target.layerId], {
      indexPattern: target.dataView,
      operations: operationDefinitionMap
    });
    return {
      ...layers,
      [target.layerId]: modifiedLayer.layer
    };
  },
  timeScalingMode: 'optional',
  paramEditor: _editor.WrappedFormulaEditor
};
exports.formulaOperation = formulaOperation;