"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.queryMappings = exports.positiveComparators = exports.negativeComparators = exports.buildFiltersFromCriteria = void 0;
var _ = require(".");
var _get_interval_in_seconds = require("../../../utils/get_interval_in_seconds");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const buildFiltersFromCriteria = (params, timestampField, executionTimeRange) => {
  const {
    timeSize,
    timeUnit,
    criteria
  } = params;
  const interval = `${timeSize}${timeUnit}`;
  const intervalAsSeconds = (0, _get_interval_in_seconds.getIntervalInSeconds)(interval);
  const intervalAsMs = intervalAsSeconds * 1000;
  const to = (executionTimeRange === null || executionTimeRange === void 0 ? void 0 : executionTimeRange.lte) || Date.now();
  const from = (executionTimeRange === null || executionTimeRange === void 0 ? void 0 : executionTimeRange.gte) || to - intervalAsMs;
  const positiveCriteria = criteria.filter(criterion => positiveComparators.includes(criterion.comparator));
  const negativeCriteria = criteria.filter(criterion => negativeComparators.includes(criterion.comparator));
  // Positive assertions (things that "must" match)
  const mustFilters = buildFiltersForCriteria(positiveCriteria);
  // Negative assertions (things that "must not" match)
  const mustNotFilters = buildFiltersForCriteria(negativeCriteria);
  const rangeFilter = {
    range: {
      [timestampField]: {
        gte: from,
        lte: to,
        format: 'epoch_millis'
      }
    }
  };

  // For group by scenarios we'll pad the time range by 1 x the interval size on the left (lte) and right (gte), this is so
  // a wider net is cast to "capture" the groups. This is to account for scenarios where we want ascertain if
  // there were "no documents" (less than 1 for example). In these cases we may be missing documents to build the groups
  // and match / not match the criteria.
  const groupedRangeFilter = {
    range: {
      [timestampField]: {
        gte: from - intervalAsMs,
        lte: to + intervalAsMs,
        format: 'epoch_millis'
      }
    }
  };
  const mustFiltersFields = positiveCriteria.map(criterion => criterion.field);
  return {
    rangeFilter,
    groupedRangeFilter,
    mustFilters,
    mustNotFilters,
    mustFiltersFields
  };
};
exports.buildFiltersFromCriteria = buildFiltersFromCriteria;
const buildFiltersForCriteria = criteria => {
  let filters = [];
  criteria.forEach(criterion => {
    const criterionQuery = buildCriterionQuery(criterion);
    if (criterionQuery) {
      filters = [...filters, criterionQuery];
    }
  });
  return filters;
};
const buildCriterionQuery = criterion => {
  const {
    field,
    value,
    comparator
  } = criterion;
  const queryType = getQueryMappingForComparator(comparator);
  switch (queryType) {
    case 'term':
      return {
        term: {
          [field]: {
            value
          }
        }
      };
    case 'match':
      {
        return {
          match: {
            [field]: value
          }
        };
      }
    case 'match_phrase':
      {
        return {
          match_phrase: {
            [field]: String(value)
          }
        };
      }
    case 'range':
      {
        const comparatorToRangePropertyMapping = {
          [_.Comparator.LT]: 'lt',
          [_.Comparator.LT_OR_EQ]: 'lte',
          [_.Comparator.GT]: 'gt',
          [_.Comparator.GT_OR_EQ]: 'gte'
        };
        const rangeProperty = comparatorToRangePropertyMapping[comparator];
        return {
          range: {
            [field]: {
              [rangeProperty]: value
            }
          }
        };
      }
    default:
      {
        return undefined;
      }
  }
};
const positiveComparators = [_.Comparator.GT, _.Comparator.GT_OR_EQ, _.Comparator.LT, _.Comparator.LT_OR_EQ, _.Comparator.EQ, _.Comparator.MATCH, _.Comparator.MATCH_PHRASE];
exports.positiveComparators = positiveComparators;
const negativeComparators = [_.Comparator.NOT_EQ, _.Comparator.NOT_MATCH, _.Comparator.NOT_MATCH_PHRASE];
exports.negativeComparators = negativeComparators;
const queryMappings = {
  [_.Comparator.GT]: 'range',
  [_.Comparator.GT_OR_EQ]: 'range',
  [_.Comparator.LT]: 'range',
  [_.Comparator.LT_OR_EQ]: 'range',
  [_.Comparator.EQ]: 'term',
  [_.Comparator.MATCH]: 'match',
  [_.Comparator.MATCH_PHRASE]: 'match_phrase',
  [_.Comparator.NOT_EQ]: 'term',
  [_.Comparator.NOT_MATCH]: 'match',
  [_.Comparator.NOT_MATCH_PHRASE]: 'match_phrase'
};
exports.queryMappings = queryMappings;
const getQueryMappingForComparator = comparator => {
  return queryMappings[comparator];
};