"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.generateCreateAgentPolicyDevToolsRequest = generateCreateAgentPolicyDevToolsRequest;
exports.generateCreatePackagePolicyDevToolsRequest = generateCreatePackagePolicyDevToolsRequest;
exports.generateUpdateAgentPolicyDevToolsRequest = generateUpdateAgentPolicyDevToolsRequest;
exports.generateUpdatePackagePolicyDevToolsRequest = generateUpdatePackagePolicyDevToolsRequest;
var _lodash = require("lodash");
var _constants = require("../../../../../../common/constants");
var _services = require("../../../services");
var _simplified_package_policy_helper = require("../../../../../../common/services/simplified_package_policy_helper");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function generateKibanaDevToolsRequest(method, path, body) {
  return `${method} kbn:${path}\n${JSON.stringify(body, null, 2)}\n`;
}

/**
 * Generate a request to create an agent policy that can be used in Kibana Dev tools
 * @param agentPolicy
 * @param withSysMonitoring
 * @returns
 */
function generateCreateAgentPolicyDevToolsRequest(agentPolicy, withSysMonitoring) {
  return generateKibanaDevToolsRequest('POST', `${_services.agentPolicyRouteService.getCreatePath()}${withSysMonitoring ? '?sys_monitoring=true' : ''}`, agentPolicy);
}

/**
 * Generate a request to create a package policy that can be used in Kibana Dev tools
 * @param packagePolicy
 * @param withSysMonitoring
 * @returns
 */
function generateCreatePackagePolicyDevToolsRequest(packagePolicy) {
  return generateKibanaDevToolsRequest('POST', _services.packagePolicyRouteService.getCreatePath(), {
    policy_id: packagePolicy.policy_id ? packagePolicy.policy_id : '<agent_policy_id>',
    package: formatPackage(packagePolicy.package),
    ...(0, _lodash.omit)(packagePolicy, 'policy_id', 'package', 'enabled'),
    inputs: formatInputs(packagePolicy.inputs),
    vars: formatVars(packagePolicy.vars)
  });
}

/**
 * Generate a request to update a package policy that can be used in Kibana Dev tools
 * @param packagePolicyId
 * @param packagePolicy
 * @returns
 */
function generateUpdatePackagePolicyDevToolsRequest(packagePolicyId, packagePolicy) {
  return generateKibanaDevToolsRequest('PUT', _services.packagePolicyRouteService.getUpdatePath(packagePolicyId), {
    package: formatPackage(packagePolicy.package),
    ...(0, _lodash.omit)(packagePolicy, 'version', 'package', 'enabled'),
    inputs: formatInputs(packagePolicy.inputs),
    vars: formatVars(packagePolicy.vars)
  });
}

/**
 * Generate a request to update an agent policy that can be used in Kibana Dev tools
 * @param agentPolicyId
 * @param agentPolicy
 * @returns
 */
function generateUpdateAgentPolicyDevToolsRequest(agentPolicyId, agentPolicy) {
  return generateKibanaDevToolsRequest('PUT', _services.agentPolicyRouteService.getUpdatePath(agentPolicyId), (0, _lodash.omit)(agentPolicy, 'version'));
}
function formatVars(vars) {
  if (!vars) {
    return;
  }
  return Object.entries(vars).reduce((acc, [varKey, varRecord]) => {
    var _varRecord$value;
    // the dataset var uses an internal format before we send it
    if (varKey === _constants.DATASET_VAR_NAME && varRecord !== null && varRecord !== void 0 && (_varRecord$value = varRecord.value) !== null && _varRecord$value !== void 0 && _varRecord$value.dataset) {
      acc[varKey] = varRecord === null || varRecord === void 0 ? void 0 : varRecord.value.dataset;
    } else {
      acc[varKey] = varRecord === null || varRecord === void 0 ? void 0 : varRecord.value;
    }
    return acc;
  }, {});
}
function formatInputs(inputs) {
  return inputs.reduce((acc, input) => {
    const inputId = (0, _simplified_package_policy_helper.generateInputId)(input);
    if (!acc) {
      acc = {};
    }
    acc[inputId] = {
      enabled: input.enabled,
      vars: formatVars(input.vars),
      streams: formatStreams(input.streams)
    };
    return acc;
  }, {});
}
function formatStreams(streams) {
  return streams.reduce((acc, stream) => {
    if (!acc) {
      acc = {};
    }
    acc[stream.data_stream.dataset] = {
      enabled: stream.enabled,
      vars: formatVars(stream.vars)
    };
    return acc;
  }, {});
}
function formatPackage(pkg) {
  return (0, _lodash.omit)(pkg, 'title');
}