"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ConnectorConfigurationLogic = void 0;
exports.dependenciesSatisfied = dependenciesSatisfied;
exports.ensureBooleanType = ensureBooleanType;
exports.ensureIntType = ensureIntType;
exports.ensureStringType = ensureStringType;
var _kea = require("kea");
var _i18n = require("@kbn/i18n");
var _is_category_entry = require("../../../../../../common/connectors/is_category_entry");
var _connectors = require("../../../../../../common/types/connectors");
var _is_not_nullish = require("../../../../../../common/utils/is_not_nullish");
var _update_connector_configuration_api_logic = require("../../../api/connector/update_connector_configuration_api_logic");
var _cached_fetch_index_api_logic = require("../../../api/index/cached_fetch_index_api_logic");
var _indices = require("../../../utils/indices");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 *
 * Sorts and filters the connector configuration
 *
 * Sorting is done by specified order (if present)
 * otherwise by alphabetic order of keys
 *
 * Filtering is done on any fields with ui_restrictions
 * or that have not had their dependencies met
 *
 */
function sortAndFilterConnectorConfiguration(config) {
  // This casting is ugly but makes all of the iteration below work for TypeScript
  // extract_full_html is only defined for crawlers, who don't use this config screen
  // we explicitly filter it out as well
  const entries = Object.entries(config).filter(([key]) => key !== 'extract_full_html');
  const groupedConfigView = entries.map(([key, entry]) => {
    if (!entry || !(0, _is_category_entry.isCategoryEntry)(entry)) {
      return null;
    }
    const configEntries = entries.map(([configKey, configEntry]) => {
      if (!configEntry || (0, _is_category_entry.isCategoryEntry)(configEntry) || configEntry.category !== key) {
        return null;
      }
      return {
        key: configKey,
        ...configEntry
      };
    }).filter(_is_not_nullish.isNotNullish);
    return {
      ...entry,
      configEntries,
      key
    };
  }).filter(_is_not_nullish.isNotNullish);
  const unCategorizedItems = filterSortValidateEntries(entries.map(([key, entry]) => entry && !(0, _is_category_entry.isCategoryEntry)(entry) && !entry.category ? {
    key,
    ...entry
  } : null).filter(_is_not_nullish.isNotNullish), config);
  const categories = groupedConfigView.map(category => {
    const configEntries = filterSortValidateEntries(category.configEntries, config);
    return configEntries.length > 0 ? {
      ...category,
      configEntries
    } : null;
  }).filter(_is_not_nullish.isNotNullish);
  return {
    categories,
    unCategorizedItems
  };
}
function filterSortValidateEntries(configEntries, config) {
  return configEntries.filter(configEntry => {
    var _configEntry$ui_restr;
    return ((_configEntry$ui_restr = configEntry.ui_restrictions) !== null && _configEntry$ui_restr !== void 0 ? _configEntry$ui_restr : []).length <= 0 && dependenciesSatisfied(configEntry.depends_on, config);
  }).sort((a, b) => {
    if ((0, _is_not_nullish.isNotNullish)(a.order)) {
      if ((0, _is_not_nullish.isNotNullish)(b.order)) {
        return a.order - b.order;
      }
      return -1;
    }
    if ((0, _is_not_nullish.isNotNullish)(b.order)) {
      // a doesn't have an order, but b has an order so takes precedence
      return 1;
    }
    return a.key.localeCompare(b.key);
  }).map(configEntry => {
    const label = configEntry.label;
    const validationErrors = [];
    if (configEntry.type === _connectors.FieldType.INTEGER && !validIntInput(configEntry.value)) {
      validationErrors.push(_i18n.i18n.translate('xpack.enterpriseSearch.content.indices.configurationConnector.config.invalidInteger', {
        defaultMessage: '{label} must be an integer.',
        values: {
          label
        }
      }));
    }
    return {
      ...configEntry,
      is_valid: validationErrors.length <= 0,
      validation_errors: validationErrors
    };
  });
}
function validIntInput(value) {
  // reject non integers (including x.0 floats), but don't validate if empty
  return (value !== null || value !== '') && (isNaN(Number(value)) || !Number.isSafeInteger(Number(value)) || ensureStringType(value).indexOf('.') >= 0) ? false : true;
}
function ensureCorrectTyping(type, value) {
  switch (type) {
    case _connectors.FieldType.INTEGER:
      return validIntInput(value) ? ensureIntType(value) : value;
    case _connectors.FieldType.BOOLEAN:
      return ensureBooleanType(value);
    default:
      return ensureStringType(value);
  }
}
function ensureStringType(value) {
  return value !== null ? String(value) : '';
}
function ensureIntType(value) {
  // int is null-safe to prevent empty values from becoming zeroes
  if (value === null || value === '') {
    return null;
  }
  return parseInt(String(value), 10);
}
function ensureBooleanType(value) {
  return Boolean(value);
}
function dependenciesSatisfied(dependencies, dependencyLookup) {
  if (!dependencies) {
    return true;
  }
  for (const dependency of dependencies) {
    var _dependencyLookup$dep;
    // casting here because this is always going to be a ConnectorConfigProperties and not a Category
    if (dependency.value !== ((_dependencyLookup$dep = dependencyLookup[dependency.field]) === null || _dependencyLookup$dep === void 0 ? void 0 : _dependencyLookup$dep.value)) {
      return false;
    }
  }
  return true;
}
const ConnectorConfigurationLogic = (0, _kea.kea)({
  actions: {
    saveConfig: true,
    setConfigState: configState => ({
      configState
    }),
    setIsEditing: isEditing => ({
      isEditing
    }),
    setLocalConfigEntry: configEntry => ({
      ...configEntry
    }),
    setLocalConfigState: configState => ({
      configState
    }),
    setShouldStartInEditMode: shouldStartInEditMode => ({
      shouldStartInEditMode
    })
  },
  connect: {
    actions: [_update_connector_configuration_api_logic.ConnectorConfigurationApiLogic, ['apiSuccess', 'makeRequest'], _cached_fetch_index_api_logic.CachedFetchIndexApiLogic, ['apiSuccess as fetchIndexApiSuccess']],
    values: [_cached_fetch_index_api_logic.CachedFetchIndexApiLogic, ['indexData as index']]
  },
  events: ({
    actions,
    values
  }) => ({
    afterMount: () => {
      actions.setConfigState((0, _indices.isConnectorIndex)(values.index) ? values.index.connector.configuration : {});
      if ((0, _indices.isConnectorIndex)(values.index) && (values.index.connector.status === _connectors.ConnectorStatus.CREATED || values.index.connector.status === _connectors.ConnectorStatus.NEEDS_CONFIGURATION)) {
        // Only start in edit mode if we haven't configured yet
        // Necessary to prevent a race condition between saving config and getting updated connector
        actions.setShouldStartInEditMode(true);
      }
    }
  }),
  listeners: ({
    actions,
    values
  }) => ({
    apiSuccess: ({
      indexName
    }) => {
      _cached_fetch_index_api_logic.CachedFetchIndexApiLogic.actions.makeRequest({
        indexName
      });
    },
    fetchIndexApiSuccess: index => {
      if (!values.isEditing && (0, _indices.isConnectorIndex)(index)) {
        actions.setConfigState(index.connector.configuration);
      }
      if (!values.isEditing && values.shouldStartInEditMode && (0, _indices.isConnectorIndex)(index) && index.connector.status === _connectors.ConnectorStatus.NEEDS_CONFIGURATION && index.connector.configuration && Object.entries(index.connector.configuration).length > 0) {
        actions.setIsEditing(true);
      }
    },
    saveConfig: () => {
      if ((0, _indices.isConnectorIndex)(values.index)) {
        actions.makeRequest({
          configuration: Object.keys(values.localConfigState).map(key => {
            var _entry$value;
            const entry = values.localConfigState[key];
            if ((0, _is_category_entry.isCategoryEntry)(entry) || !entry) {
              return null;
            }
            return {
              key,
              value: (_entry$value = entry.value) !== null && _entry$value !== void 0 ? _entry$value : ''
            };
          }).filter(_is_not_nullish.isNotNullish).reduce((prev, {
            key,
            value
          }) => {
            prev[key] = value;
            return prev;
          }, {}),
          connectorId: values.index.connector.id,
          indexName: values.index.connector.index_name
        });
      }
    },
    setIsEditing: isEditing => {
      if (isEditing) {
        actions.setLocalConfigState(values.configState);
      }
    }
  }),
  path: ['enterprise_search', 'content', 'connector_configuration'],
  reducers: () => ({
    configState: [{}, {
      apiSuccess: (_, {
        configuration
      }) => configuration,
      setConfigState: (_, {
        configState
      }) => configState
    }],
    isEditing: [false, {
      apiSuccess: () => false,
      setIsEditing: (_, {
        isEditing
      }) => isEditing
    }],
    localConfigState: [{}, {
      setLocalConfigEntry: (configState, {
        key,
        display,
        type,
        validations,
        value,
        ...configEntry
      }) => ({
        ...configState,
        [key]: {
          ...configEntry,
          display,
          type,
          validations: validations !== null && validations !== void 0 ? validations : [],
          value: display ? ensureCorrectTyping(type, value) : value // only check type if field had a specified eui element
        }
      }),

      setLocalConfigState: (_, {
        configState
      }) => configState
    }],
    shouldStartInEditMode: [false, {
      apiSuccess: () => false,
      setShouldStartInEditMode: (_, {
        shouldStartInEditMode
      }) => shouldStartInEditMode
    }]
  }),
  selectors: ({
    selectors
  }) => ({
    configView: [() => [selectors.configState], configState => sortAndFilterConnectorConfiguration(configState)],
    localConfigView: [() => [selectors.localConfigState], configState => sortAndFilterConnectorConfiguration(configState)]
  })
});
exports.ConnectorConfigurationLogic = ConnectorConfigurationLogic;