"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseModelStateReasonFromStats = exports.parseModelStateFromStats = exports.parseMlInferenceParametersFromPipeline = exports.getSetProcessorForInferenceType = exports.getRemoveProcessorForInferenceType = exports.getMlModelTypesForModelConfig = exports.getMlInferencePrefixedFieldName = exports.getInferenceProcessor = exports.generateMlInferencePipelineBody = exports.formatPipelineName = exports.TEXT_EXPANSION_TYPE = exports.TEXT_EXPANSION_FRIENDLY_TYPE = exports.ML_INFERENCE_PREFIX = exports.ELSER_MODEL_ID = void 0;
var _mlTrainedModelsUtils = require("@kbn/ml-trained-models-utils");
var _pipelines = require("../types/pipelines");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TEXT_EXPANSION_TYPE = _mlTrainedModelsUtils.SUPPORTED_PYTORCH_TASKS.TEXT_EXPANSION;
exports.TEXT_EXPANSION_TYPE = TEXT_EXPANSION_TYPE;
const TEXT_EXPANSION_FRIENDLY_TYPE = 'ELSER';
exports.TEXT_EXPANSION_FRIENDLY_TYPE = TEXT_EXPANSION_FRIENDLY_TYPE;
const ML_INFERENCE_PREFIX = 'ml.inference.';
exports.ML_INFERENCE_PREFIX = ML_INFERENCE_PREFIX;
const ELSER_MODEL_ID = '.elser_model_1';
exports.ELSER_MODEL_ID = ELSER_MODEL_ID;
/**
 * Generates the pipeline body for a machine learning inference pipeline
 * @param pipelineConfiguration machine learning inference pipeline configuration parameters
 * @returns pipeline body
 */
const generateMlInferencePipelineBody = ({
  description,
  fieldMappings,
  inferenceConfig,
  model,
  pipelineName
}) => {
  const inferenceType = Object.keys(model.inference_config || {})[0];
  const pipelineDefinition = {
    description: description !== null && description !== void 0 ? description : '',
    processors: [],
    version: 1
  };
  pipelineDefinition.processors = [
  // Add remove and inference processors
  ...fieldMappings.flatMap(({
    sourceField,
    targetField
  }) => {
    const remove = getRemoveProcessorForInferenceType(targetField, inferenceType);
    const inference = getInferenceProcessor(sourceField, targetField, inferenceConfig, model, pipelineName);
    return [{
      remove: {
        field: getMlInferencePrefixedFieldName(targetField),
        ignore_missing: true
      }
    }, ...(remove ? [{
      remove
    }] : []), {
      inference
    }];
  }),
  // Add single append processor
  {
    append: {
      field: '_source._ingest.processors',
      value: [{
        model_version: model.version,
        pipeline: pipelineName,
        processed_timestamp: '{{{ _ingest.timestamp }}}',
        types: getMlModelTypesForModelConfig(model)
      }]
    }
  },
  // Add set processors
  ...fieldMappings.flatMap(({
    targetField
  }) => {
    const set = getSetProcessorForInferenceType(targetField, inferenceType);
    return set ? [{
      set
    }] : [];
  })];
  return pipelineDefinition;
};
exports.generateMlInferencePipelineBody = generateMlInferencePipelineBody;
const getInferenceProcessor = (sourceField, targetField, inferenceConfig, model, pipelineName) => {
  var _model$input, _model$input$field_na;
  // If model returned no input field, insert a placeholder
  const modelInputField = ((_model$input = model.input) === null || _model$input === void 0 ? void 0 : (_model$input$field_na = _model$input.field_names) === null || _model$input$field_na === void 0 ? void 0 : _model$input$field_na.length) > 0 ? model.input.field_names[0] : 'MODEL_INPUT_FIELD';
  return {
    field_map: {
      [sourceField]: modelInputField
    },
    inference_config: inferenceConfig,
    model_id: model.model_id,
    on_failure: [{
      append: {
        field: '_source._ingest.inference_errors',
        value: [{
          message: `Processor 'inference' in pipeline '${pipelineName}' failed with message '{{ _ingest.on_failure_message }}'`,
          pipeline: pipelineName,
          timestamp: '{{{ _ingest.timestamp }}}'
        }]
      }
    }],
    target_field: getMlInferencePrefixedFieldName(targetField)
  };
};
exports.getInferenceProcessor = getInferenceProcessor;
const getSetProcessorForInferenceType = (targetField, inferenceType) => {
  let set;
  if (inferenceType === _mlTrainedModelsUtils.SUPPORTED_PYTORCH_TASKS.TEXT_CLASSIFICATION) {
    set = {
      copy_from: `${getMlInferencePrefixedFieldName(targetField)}.predicted_value`,
      description: `Copy the predicted_value to '${targetField}' if the prediction_probability is greater than 0.5`,
      field: targetField,
      if: `ctx?.ml?.inference != null && ctx.ml.inference['${targetField}'] != null && ctx.ml.inference['${targetField}'].prediction_probability > 0.5`,
      value: undefined
    };
  } else if (inferenceType === _mlTrainedModelsUtils.SUPPORTED_PYTORCH_TASKS.TEXT_EMBEDDING) {
    set = {
      copy_from: `${getMlInferencePrefixedFieldName(targetField)}.predicted_value`,
      description: `Copy the predicted_value to '${targetField}'`,
      field: targetField,
      if: `ctx?.ml?.inference != null && ctx.ml.inference['${targetField}'] != null`,
      value: undefined
    };
  }
  return set;
};
exports.getSetProcessorForInferenceType = getSetProcessorForInferenceType;
const getRemoveProcessorForInferenceType = (targetField, inferenceType) => {
  if (inferenceType === _mlTrainedModelsUtils.SUPPORTED_PYTORCH_TASKS.TEXT_CLASSIFICATION || inferenceType === _mlTrainedModelsUtils.SUPPORTED_PYTORCH_TASKS.TEXT_EMBEDDING) {
    return {
      field: targetField,
      ignore_missing: true
    };
  }
};

/**
 * Parses model types list from the given configuration of a trained machine learning model
 * @param trainedModel configuration for a trained machine learning model
 * @returns list of model types
 */
exports.getRemoveProcessorForInferenceType = getRemoveProcessorForInferenceType;
const getMlModelTypesForModelConfig = trainedModel => {
  var _trainedModel$tags;
  if (!trainedModel) return [];
  const isBuiltIn = (_trainedModel$tags = trainedModel.tags) === null || _trainedModel$tags === void 0 ? void 0 : _trainedModel$tags.includes(_mlTrainedModelsUtils.BUILT_IN_MODEL_TAG);
  return [trainedModel.model_type, ...Object.keys(trainedModel.inference_config || {}), ...(isBuiltIn ? [_mlTrainedModelsUtils.BUILT_IN_MODEL_TAG] : [])].filter(type => type !== undefined);
};
exports.getMlModelTypesForModelConfig = getMlModelTypesForModelConfig;
const formatPipelineName = rawName => rawName.trim().replace(/\s+/g, '_') // Convert whitespaces to underscores
.toLowerCase();
exports.formatPipelineName = formatPipelineName;
const parseMlInferenceParametersFromPipeline = (name, pipeline) => {
  var _pipeline$processors, _inferenceProcessor$f;
  const processor = pipeline === null || pipeline === void 0 ? void 0 : (_pipeline$processors = pipeline.processors) === null || _pipeline$processors === void 0 ? void 0 : _pipeline$processors.find(proc => proc.inference !== undefined);
  if (!processor || (processor === null || processor === void 0 ? void 0 : processor.inference) === undefined) {
    return null;
  }
  const {
    inference: inferenceProcessor
  } = processor;
  const sourceFields = Object.keys((_inferenceProcessor$f = inferenceProcessor.field_map) !== null && _inferenceProcessor$f !== void 0 ? _inferenceProcessor$f : {});
  const sourceField = sourceFields.length === 1 ? sourceFields[0] : null;
  if (!sourceField) {
    return null;
  }
  return {
    destination_field: inferenceProcessor.target_field ? stripMlInferencePrefix(inferenceProcessor.target_field) : inferenceProcessor.target_field,
    model_id: inferenceProcessor.model_id,
    pipeline_name: name,
    source_field: sourceField
  };
};
exports.parseMlInferenceParametersFromPipeline = parseMlInferenceParametersFromPipeline;
const parseModelStateFromStats = (model, modelTypes) => {
  var _model$deployment_sta;
  if ((model === null || model === void 0 ? void 0 : model.model_type) === _mlTrainedModelsUtils.TRAINED_MODEL_TYPE.LANG_IDENT || modelTypes !== null && modelTypes !== void 0 && modelTypes.includes(_mlTrainedModelsUtils.TRAINED_MODEL_TYPE.LANG_IDENT)) return _pipelines.TrainedModelState.Started;
  switch (model === null || model === void 0 ? void 0 : (_model$deployment_sta = model.deployment_stats) === null || _model$deployment_sta === void 0 ? void 0 : _model$deployment_sta.state) {
    case 'started':
      return _pipelines.TrainedModelState.Started;
    case 'starting':
      return _pipelines.TrainedModelState.Starting;
    case 'stopping':
      return _pipelines.TrainedModelState.Stopping;
    // @ts-ignore: type is wrong, "failed" is a possible state
    case 'failed':
      return _pipelines.TrainedModelState.Failed;
    default:
      return _pipelines.TrainedModelState.NotDeployed;
  }
};
exports.parseModelStateFromStats = parseModelStateFromStats;
const parseModelStateReasonFromStats = trainedModelStats => {
  var _trainedModelStats$de;
  return trainedModelStats === null || trainedModelStats === void 0 ? void 0 : (_trainedModelStats$de = trainedModelStats.deployment_stats) === null || _trainedModelStats$de === void 0 ? void 0 : _trainedModelStats$de.reason;
};
exports.parseModelStateReasonFromStats = parseModelStateReasonFromStats;
const getMlInferencePrefixedFieldName = fieldName => fieldName.startsWith(ML_INFERENCE_PREFIX) ? fieldName : `${ML_INFERENCE_PREFIX}${fieldName}`;
exports.getMlInferencePrefixedFieldName = getMlInferencePrefixedFieldName;
const stripMlInferencePrefix = fieldName => fieldName.startsWith(ML_INFERENCE_PREFIX) ? fieldName.replace(ML_INFERENCE_PREFIX, '') : fieldName;