"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useRowHeightsOptions = void 0;
var _react = require("react");
var _common = require("../../common");
var _validate_row_height = require("../utils/validate_row_height");
var _use_discover_services = require("./use_discover_services");
var _row_heights = require("../utils/row_heights");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Row height might be a value from -1 to 20
 * A value of -1 automatically adjusts the row height to fit the contents.
 * A value of 0 displays the content in a single line.
 * A value from 1 to 20 represents number of lines of Document explorer row to display.
 */
const SINGLE_ROW_HEIGHT_OPTION = 0;
const AUTO_ROW_HEIGHT_OPTION = -1;

/**
 * Converts rowHeight of EuiDataGrid to rowHeight number (-1 to 20)
 */
const serializeRowHeight = rowHeight => {
  if (rowHeight === 'auto') {
    return AUTO_ROW_HEIGHT_OPTION;
  } else if (typeof rowHeight === 'object' && rowHeight.lineCount) {
    return rowHeight.lineCount; // custom
  }

  return SINGLE_ROW_HEIGHT_OPTION;
};

/**
 * Converts rowHeight number (-1 to 20) of EuiDataGrid rowHeight
 */
const deserializeRowHeight = number => {
  if (number === AUTO_ROW_HEIGHT_OPTION) {
    return 'auto';
  } else if (number === SINGLE_ROW_HEIGHT_OPTION) {
    return undefined;
  }
  return {
    lineCount: number
  }; // custom
};

const useRowHeightsOptions = ({
  rowHeightState,
  onUpdateRowHeight
}) => {
  const {
    storage,
    uiSettings
  } = (0, _use_discover_services.useDiscoverServices)();
  return (0, _react.useMemo)(() => {
    const rowHeightFromLS = (0, _row_heights.getStoredRowHeight)(storage);
    const configRowHeight = uiSettings.get(_common.ROW_HEIGHT_OPTION);
    const configHasNotChanged = localStorageRecord => localStorageRecord !== null && configRowHeight === localStorageRecord.previousConfigRowHeight;
    let rowHeight;
    if ((0, _validate_row_height.isValidRowHeight)(rowHeightState)) {
      rowHeight = rowHeightState;
    } else if (configHasNotChanged(rowHeightFromLS)) {
      rowHeight = rowHeightFromLS.previousRowHeight;
    } else {
      rowHeight = configRowHeight;
    }
    return {
      defaultHeight: deserializeRowHeight(rowHeight),
      lineHeight: '1.6em',
      onChange: ({
        defaultHeight: newRowHeight
      }) => {
        const newSerializedRowHeight = serializeRowHeight(newRowHeight);
        (0, _row_heights.updateStoredRowHeight)(newSerializedRowHeight, configRowHeight, storage);
        onUpdateRowHeight === null || onUpdateRowHeight === void 0 ? void 0 : onUpdateRowHeight(newSerializedRowHeight);
      }
    };
  }, [rowHeightState, uiSettings, storage, onUpdateRowHeight]);
};
exports.useRowHeightsOptions = useRowHeightsOptions;